;;; jde-db.el -- Debugger mode for jdb.
;; $Revision: 1.11 $ $Date: 1997/10/26 05:49:59 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Version: 1.8
;; Keywords: java, tools

;; Copyright (C) 1997 Free Software Foundation, Inc.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, US
;;; Commentary:

;; This package interfaces emacs to jdb, the debugger
;; distributed as part of JavaSoft's Java
;; Development Kit (JDK).

;; Please send bug reports and enhancement suggestions
;; to Paul Kinnucan at <paulk@mathworks.com>

;; Change History
;; $Log: jde-db.el $
;; Revision 1.11  1997/10/26 05:49:59  kinnucan
;; Applied Derek Young's patch to cause jde to qualify class names
;; when setting a breakpoint.
;;
;; Revision 1.10  1997/10/20 05:27:48  kinnucan
;; Removed reference to deleted function jde-db-massage-args
;;
;; Revision 1.9  1997/10/11 01:36:05  kinnucan
;; Fixed bug in jde-db-search-src-dirs discovered by Jonathan Payne.
;;
;; Revision 1.8  1997/10/06 14:40:53  kinnucan
;; Fixed bugs in jde-db-set-debugger command.
;;
;; Revision 1.7  1997/10/05 21:20:15  kinnucan
;; 1. Added the variables jde-db-debugger and jde-db-debugger-is-executable
;;    and the associated setter function jde-db-set-debugger. These allow
;;    you to specify a custom debugger for the JDE>
;;
;; 2. Added jde-db-args and jde-db-app-args and the associated setter
;;    functions. These allow you to specify debugger and application
;;    command line arguments.
;;
;; Revision 1.6  1997/10/05 04:53:04  kinnucan
;; Fixed bug in print object menu item.
;;
;; Revision 1.5  1997/08/26 14:53:39  paulk
;; Fixed bug in check-source-path.
;;
;; Revision 1.4  1997/08/26 08:52:14  kinnucan
;; Tweaked JDE Version number for JDE 1.8 release.
;;
;; Revision 1.3  1997/07/05 04:18:10  kinnucan
;; Updated make-jdb-command to run either the class previously specifed with
;; the jde-run-set-app command or the class corresponding to the code in the
;; current buffer.
;;
;; Revision 1.2  1997/06/18 18:45:11  paulk
;; Added error-checking to jde-db-set-source-paths function. Now checks for
;; existence of specified directories and appends a terminal slash to paths
;; that lack it.
;;
;; Revision 1.1  1997/06/18 17:21:59  paulk
;; Initial revision
;;

;;; Code:

(require 'gud)

;; ======================================================================
;; jde-db functions

;;; History of argument lists passed to jde-db.
(defvar jde-db-history nil)

(defvar jde-db-debugger "jdb"
"Path name of debugger, if debugger ia an executable; otherwise,
the fully qualified package name of the debugger class. Default is jdb.")

(defvar jde-db-debugger-is-executable t
"t if debugger is an executable file; if nil, assume that the
debugger is a Java class file. Default is t.")

(defvar jde-db-args nil
"List of debugger arguments.")

(defvar jde-db-app-args nil
"List of arguments to pass to the application via the command line.")

(defun jde-db-set-debugger (debugger is-executable)
  "Specify the pathname of the debugger, if an executable, or the
debugger's fully qualified class name, if a class."
  (interactive
   "sEnter name of Java interpreter: \nsIs %s executable? (yes): ")
  (setq jde-db-debugger debugger)
  (if (stringp is-executable)
      (if (or
	   (string= is-executable "")
	   (eq (aref is-executable 0) ?y))
	  (setq jde-db-debugger-is-executable t)
	(setq jde-db-debugger-is-executable nil))
    (setq jde-db-debugger-is-executable is-executable)))

(defun jde-db-set-args (args)
  "Specify the arguments (except -classpath) to be passed to the debugger."
  (interactive 
   "sEnter arguments: ")
  (setq jde-db-args (jde-run-parse-args args)))
  
(defun jde-db-make-debugger-arg-string ()
 (let ((classpath-arg (jde-get-classpath-arg))
       (other-args (jde-run-make-arg-string jde-db-args)))
   (if classpath-arg
       (setq classpath-arg (concat classpath-arg " ")))
   (if other-args
       (setq other-args (concat other-args " ")))
   (concat classpath-arg other-args)))

(defun jde-db-set-app-args (args)
  "Specify the arguments to be passed to the Java application class."
  (interactive 
   "sEnter arguments: ")
  (setq jde-db-app-args (jde-run-parse-args args)))


(defvar jde-db-source-directories (list "d:/jdk1.1/src/")
  "*List of source directory paths.
jde-db uses this list to locate source files visited by the debugger
while stepping through an application.")


(defvar jde-db-marker-regexp
  "(\\(.*\\):\\([0-9]*\\))")

(defvar jde-db-qualified-class-name-regexp
"\\( \\|[^ ]*\\.\\)\\([^ .]*\\.[^ .]* \\)")

(defvar jde-db-last-package ()
  "Package that the debugger is currently visiting.")

(defvar jde-db-xemacs-menu
  '(["Continue"          gud-cont t]
    ["Next Line"         gud-next t]
    ["Step Line"         gud-step t]
    ["Print"             gud-print t]
    ["Down Stack"        gud-down t]
    ["Up Stack"          gud-up t]
    ["Set Breakpoint"    gud-break t]
    ["Remove Breakpoint" gud-remove t]
    )
  "XEmacs 19 menu for java debugger.")

(defun jde-db-xemacs-menu ()
  (cons "Jdb" jde-db-xemacs-menu))

(defun jde-db-marker-filter (input)
  (let ((output input))
    
    ;; Check for a jdb stepper message.
    (if (string-match jde-db-marker-regexp input)
	;; Extract the frame position from the marker.
	(let ((marker (substring input (match-beginning 0) (match-end 0)))
	      (class (substring input (match-beginning 1) (match-end 1)))
	      (line-no (string-to-int (substring input
						 (match-beginning 2)
						 (match-end 2))))
	      (rest (substring input (match-end 0))))

	  (setq gud-last-frame (cons (concat class ".java") line-no))

	  ;; Extract package path from input.
	  (string-match jde-db-qualified-class-name-regexp input)
	  (setq jde-db-last-package
		(substring output (match-beginning 1) (match-end 1)))

	  (setq output (concat "Stopped at " 
			       (substring marker 1
					  (- (length marker) 1)) rest))))

    output))

(defun jde-db-contains-file-p (dir file)
  "Return t if DIR contains FILE."
  (let ((files (directory-files dir)))
    (catch 'found
      (let ((n (- (length files) 1)))
	(while (>= n 0)
	  (if (string= file (elt files n))
	      (throw 'found t))
	  (setq n (- n 1)))))))

(defun jde-db-contains-package-p (dir package)
  "Return t if DIR contains PACKAGE."
  (let ((files (directory-files dir))
	(pkg-root (substring package
			     0
			     (string-match "\\." package))))
    (catch 'found
      (let ((n (- (length files) 1)))
	(while (>= n 0)
	  (let ((curr-file (elt files n)))
	    (if (string= pkg-root curr-file)
		(throw 'found t)))
	  (setq n (- n 1)))))))

(defun jde-db-pkg-to-path (package)
  "Return PACKAGE as a directory path."
  (let ((n (string-match "\\." package))
	(output (concat package)))
    (while n
      (aset output n ?/)
      (setq n (string-match "\\." output (+ n 1))))
    output))
      

(defun jde-db-search-src-dirs (file)
  "Return the directory containing FILE."
  (catch 'found
    (let ((n (- (length jde-db-source-directories) 1)))
      (while (>= n 0)
	(let ((curr-dir (elt jde-db-source-directories n)))
	  (cond
	   ((jde-db-contains-file-p curr-dir file)
	    (throw 'found curr-dir))
	   ((and (jde-db-contains-package-p curr-dir jde-db-last-package)
		 (jde-db-contains-file-p
		  (concat curr-dir (jde-db-pkg-to-path jde-db-last-package)) file))
	    (throw 'found
		   (concat curr-dir 
			   (jde-db-pkg-to-path jde-db-last-package))))))
	(setq n (- n 1))))))


;; Fixes a bug in gud-make-debug-menu
(defun jde-db-make-debug-menu ()
  "Make sure the current local map has a [menu-bar debug] submap.
If it doesn't, replace it with a new map that inherits it,
and create such a submap in that new map."
  (if (and (current-local-map)
	   (lookup-key (current-local-map) [menu-bar debug]))
      nil
    (use-local-map (gud-new-keymap (current-local-map)))
    (define-key (current-local-map) [menu-bar debug]
      (nconc (list "Jdb") gud-menu-map))))



(defun jde-db-find-file (f)
  (save-excursion
    (let ((buf (find-file-noselect 
		(concat (jde-db-search-src-dirs f) f))))
      (set-buffer buf)
      (if (string-match "XEmacs\\|Lucid" emacs-version)
	  (if (and (boundp 'current-menubar)
		   current-menubar)
	      (if (fboundp 'add-submenu)
		  (add-submenu nil (jde-db-xemacs-menu))
		(add-menu nil "Jdb" jde-deb-xemacs-menu)))
	(progn
	  (jde-db-make-debug-menu)
	  (local-set-key [menu-bar debug refresh] nil)
	  (local-set-key [menu-bar debug stepi] nil)
	  (local-set-key [menu-bar debug up] '("Up Stack" . gud-up))
	  (local-set-key [menu-bar debug down] '("Down Stack" . gud-down))))
      buf)))

(defvar jde-db-minibuffer-local-map nil
  "Keymap for minibuffer prompting of jdb startup command.")
(if jde-db-minibuffer-local-map
    ()
  (setq jde-db-minibuffer-local-map (copy-keymap minibuffer-local-map))
  (define-key
    jde-db-minibuffer-local-map "\C-i" 'comint-dynamic-complete-filename))

(defvar jde-db-mode-hook nil
  "Customization hook for jde-db inferior mode.")

(defun class-from-file-name (file-name)
  (file-name-sans-extension (file-name-nondirectory file-name)))

;;; The jde-db-call function must do the right thing whether its invoking
;;; keystroke is from the GUD buffer itself (via major-mode binding)
;;; or a C buffer.  In the former case, we want to supply data from
;;; gud-last-frame.  Here's how we do it:
;;; Note: this is adapted from the gud-format-command function
;;; in gud.el.

(defun jde-db-format-command (str arg)
  (let ((insource (not (eq (current-buffer) gud-comint-buffer)))
	(frame (or gud-last-frame gud-last-last-frame))
	result)
    (while (and str (string-match "\\([^%]*\\)%\\([acdeflp]\\)" str))
      (let ((key (string-to-char (substring str (match-beginning 2))))
	    (group1 (substring str (match-beginning 1) (match-end 1)))
	    subst)
	(setq str (substring str (match-end 2)))
	(cond
	 ((eq key ?f)
	  (setq subst (file-name-nondirectory (if insource
						  (buffer-file-name)
						(car frame)))))
	 ((eq key ?c)
	  (setq subst (concat (jde-db-get-package)
			      (class-from-file-name (if insource
							(buffer-file-name)
						      (car frame))))))

	 ((eq key ?d)
	  (setq subst (file-name-directory (if insource
					       (buffer-file-name)
					     (car frame)))))
	 ((eq key ?l)
	  (setq subst (if insource
			  (save-excursion
			    (beginning-of-line)
			    (save-restriction (widen)
					      (1+ (count-lines 1 (point)))))
			(cdr frame))))
	 ((eq key ?e)
	  (setq subst (find-c-expr)))
	 ((eq key ?a)
	  (setq subst (gud-read-address)))
	 ((eq key ?p)
	  (setq subst (if arg (int-to-string arg) ""))))
	(setq result (concat result group1 subst))))
    ;; There might be text left in STR when the loop ends.
    (concat result str)))

(defun jde-db-call (fmt &optional arg)
  (let ((msg (jde-db-format-command fmt arg)))
    (message "Command: %s" msg)
    (sit-for 0)
    (gud-basic-call msg)))

(defmacro jde-db-def (func cmd key &optional doc)
  "Define FUNC to be a command sending CMD and bound to KEY, with
optional doc string DOC.  Certain %-escapes in the string arguments
are interpreted specially if present.  These are:

  %f name (without directory) of current source file.
  %c    fully qualified class name
  %d directory of current source file.
  %l number of current source line
  %e text of the C lvalue or function-call expression surrounding point.
  %a text of the hexadecimal address surrounding point
  %p prefix argument to the command (if any) as a number

  The `current' source file is the file of the current buffer (if
we're in a C file) or the source file current at the last break or
step (if we're in the GUD buffer).
  The `current' line is that of the current buffer (if we're in a
source file) or the source line number at the last break or step (if
we're in the GUD buffer)."
  (list 'progn
	(list 'defun func '(arg)
	      (or doc "")
	      '(interactive "p")
	      (list 'jde-db-call cmd 'arg))
	(if key
	    (list 'define-key
		  '(current-local-map)
		  (concat "\C-c" key)
		  (list 'quote func)))
	(if key
	    (list 'global-set-key
		  (list 'concat 'gud-key-prefix key)
		  (list 'quote func)))))

(defun jde-db-init(app-class marker-filter find-file)
  (let ((debug-buf-name (concat "*debug" app-class "*")))
    (if (not (comint-check-proc debug-buf-name))
	(let* ((debug-buffer (get-buffer-create debug-buf-name))
	       (program (if jde-db-debugger-is-executable
			   jde-db-debugger
			 jde-run-java-vm))
	       (prog-args (if jde-db-debugger-is-executable
			      (append 
			       (if (not
				    (string= jde-classpath ""))
				   (list "-classpath" 
					 (jde-run-unquote jde-classpath)))
			       jde-db-args
			       (list app-class)
			       jde-db-app-args)
			    (append
			     (list jde-db-debugger)
			     (if (not
				  (string= jde-classpath ""))
				 (list "-classpath"
				       (jde-run-unquote jde-classpath)))
			     jde-db-args
			     (list app-class)
			     jde-db-app-args)))
	       (command-string (concat
				program " "
				(jde-run-make-arg-string prog-args) "\n\n")))
	  (save-excursion
	    (set-buffer debug-buffer)
	    (erase-buffer)
	    (insert (concat "cd " default-directory "\n"))
	    (insert command-string)
	    (comint-mode))
	  (comint-exec debug-buffer app-class program nil prog-args)
	  (pop-to-buffer debug-buffer)
	  (gud-mode)
	  (make-local-variable 'gud-marker-filter)
	  (setq gud-marker-filter marker-filter)
	  (make-local-variable 'gud-find-file)
	  (setq gud-find-file find-file)
	  (set-process-filter (get-buffer-process (current-buffer)) 'gud-filter)
	  (set-process-sentinel (get-buffer-process (current-buffer)) 'gud-sentinel)
	  (gud-set-buffer))
      (message "An instance of %s is running." app-class)			
      (pop-to-buffer debug-buf-name))))			   
	  


(defun jde-db-get-package ()
  "Return the package of the class whose source file resides in the current
buffer."
  (save-excursion
    (goto-char (point-min))
    (if (re-search-forward "\\(package\\) *\\([^ ]*\\) *;" (point-max) t)
	(concat (buffer-substring-no-properties (match-beginning 2) (match-end 2))
		"."))))

(defun make-jdb-command ()
  (let ((app-class (if jde-run-application-class
		       jde-run-application-class
		     (concat (jde-db-get-package)
			     (file-name-sans-extension 
			      (file-name-nondirectory (buffer-file-name)))))))
    (concat "jdb " app-class)))
		   
(defun jde-db ()
  "Run jdb on Java class whose source resides in the current buffer.
This command creates a command buffer named *gud-CLASS* for the debug session.
CLASS is the name of the class you are debugging."
  (interactive)

  ;; test for XEmacs
  (if (string-match "XEmacs\\|Lucid" emacs-version)
      (gud-overload-functions
       '((gud-marker-filter . jde-db-marker-filter)
	 (gud-find-file . jde-db-find-file)
	 )))

  (let ((app-class jde-run-application-class))
    (if (not app-class)
	(setq app-class
	      (concat (jde-db-get-package)
		      (file-name-sans-extension 
		       (file-name-nondirectory (buffer-file-name))))))
    (jde-db-init app-class 'jde-db-marker-filter 'jde-db-find-file))

  (jde-db-def gud-break  "stop at %c:%l"  "\C-b" "Set breakpoint at current line.")
  (jde-db-def gud-remove "clear %c:%l"    "\C-d" "Remove breakpoint at current line")
  (jde-db-def gud-step   "step"           "\C-s" "Step one source line with display.")
  (jde-db-def gud-next   "next"           "\C-n" "Step one line (skip functions).")
  (jde-db-def gud-cont   "cont"           "\C-r" "Continue with display.")
  (jde-db-def gud-up     "up %p"          "<" "Up N stack frames (numeric arg).")
  (jde-db-def gud-down   "down %p"        ">" "Down N stack frames (numeric arg).")
  (jde-db-def gud-print  "print %e"       "\C-p" "Print object.")
  (local-set-key [menu-bar debug up]   '("Up Stack" . gud-up))
  (local-set-key [menu-bar debug down] '("Down Stack" . gud-down))
  (setq comint-prompt-regexp "\\(^> *\\)\\|\\(^.*\\[[0-9]*\\] *\\)")
  (setq paragraph-start comint-prompt-regexp)
  (run-hooks 'jde-db-mode-hook)
  )

(defun check-source-path (path) 
  "Return a valid path or nil if path is not found."  
  ;; Ensure that path ends in a slash.
  (let ((p (if (not (string= (substring path (- (length path) 1)) "/"))
	       (concat path "/")
	     path)))
    (if (file-directory-p p)
	p
      nil)))

(defun jde-db-set-source-paths (paths)
 "Set the source path list used by JDE to locate Java source files visited
by the debugger. PATHS is a list of source paths separated by colons or semicolons,
depending on the operating system.. Note that for packages, you need enter only the 
directory containing the top-level package. For example, including '../jdk1.1/src/'
 enables the JDE to  locate all source files provided with JDK1.1. Note also that the 
paths must end in a slash."
 (interactive
  "sEnter source paths: ")
 (let ((m 0)
       (n (string-match path-separator paths)))
   (setq jde-db-source-directories (list))
   (while n
     (let ((path (check-source-path (substring paths m n))))
       (if path
	   (setq jde-db-source-directories
		 (cons path jde-db-source-directories)))
       (setq m (+ n 1))
       (setq n (string-match path-separator paths m))))
   (setq n (length paths))
   (if (and (> n 0) (< m n))
       (let ((path (check-source-path (substring paths m n))))
	 (if path
	     (setq jde-db-source-directories
		   (cons (substring paths m n) jde-db-source-directories)))))
   (setq jde-db-source-directories (nreverse jde-db-source-directories))))


(provide 'jde-db)

;;; end of jde-db.el
