;;; jde.el -- Integrated Development Environment for Java.
;; $Revision: 1.13 $ $Date: 1997/10/20 05:21:20 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1997 Free Software Foundation, Inc.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This package turns Emacs into an integrated development
;; environment for Java. It provides the following
;; capabilities:
;;
;;   -- Specialized editing mode for Java source code
;;      (via cc-mode.el)
;;
;;   -- Java tools commands
;;
;;   -- Compile a Java app
;;
;;   -- Run App (runs a Java app in a comint buffer)
;;
;;   -- Debug  (jdb, runs debugger)
;;
;;   -- Browse Doc (runs browser on JDK 1.x API doc)
;;
;;   -- Java debug mode
;;     
;; This mode displays source file being debugged
;; with pointer to current line.
;;
;; This package includes:
;;
;;    -- jde-run.el
;;
;;    -- jde-db.el
;; 
;;    -- speedbar.el
;;
;;
;;  This package requires:
;;
;;    -- latest version of cc-mode.el
;;
;;       (available from
;;	 ftp://ftp.python.org/pub/emacs/cc-mode.tar.gz)
;;
;;    -- andersl-java-font-lock.el for Emacs versions < 20
;;
;;       (available from 
;;        http:\\www.csd.uu.se\~andersl\emacs.html
;;        ftp:\\ftp.csd.uu.se\pub\users\andersl\emacs\)
;;
;;
;;       This is required for syntax highlighting.
;;
;;    -- JavaSoft's Java Development Kit (jdk) 
;;     
;;       (available from
;;	 http://www.javasoft.com/products/jdk/1.1/index.html)
;;
;;       -- Web browser
;;
;;       (e.g., Netscape or Internet Explorer)
;;
;;    -- Unix-style shell for Windows 95 or Windows/NT
;;
;;    I recommend bash, available from www.cygnus.com.
;;
;; To install this package:
;;
;; 1. Copy jde.el, jde-db.el, jde-run.el, and speedbar.el files
;;    to a directory in your emacs lisp file load
;;    path.
;;
;; 2. Byte-compile these files if you like.
;;
;; 3. Add the following code to your .emacs file:
;;
;;    (load "jde")
;;    (setq jde-web-browser "BROWSER")
;;    (setq jde-doc-dir "JDK DIRECTORY")
;;
;; For example, suppose that Netscape is your
;; browser and is in your shell command path
;; and you have installed the JDK API doc at
;; c:/JDK1.1/docs. Then, you would add
;;
;;    (load "jde")
;;    (setq jde-web-browser "netscape")
;;    (setq jde-doc-dir "c:/jdk1.1/docs/")
;;
;; to your .emacs file.
;;
;; If you are using the bash shell, you should 
;; specify this either by setting the SHELL
;; environment variable or the elisp variable
;; shell-file-name to bash. For example, for
;; Win95, enter
;;   
;;    set shell=bash
;;
;; in your autoexec.bat file or
;;
;;    (setq shell-file-name "bash")
;;
;; in your .emacs file.
;;
;; If you are using bash on Windows 95 or Windows/NT, you should
;; also put 
;;
;;    (setq shell-command-switch "-c")
;;
;; in your .emacs file.
;;
;; 4. Add the JDK bin directory to your shell's
;;    command path, if necessary
;;
;; 5. Set your shell's Java CLASSPATH environment variable so
;;    that it includes the path to your Java
;;    app code.
;;
;; Please send any comments, bugs, or upgrade requests to
;; Paul Kinnucan at paulk@mathworks.com.

;; Change History
;;
;; $Log: jde.el $
;; Revision 1.13  1997/10/20 05:21:20  kinnucan
;; Now requires andersl-java-font-lock only for Emacs versions < 20.
;;
;; Revision 1.12  1997/10/18 05:24:52  kinnucan
;; 1. Changed key bindings to use the two prefix keys C-c C-v.
;;
;; 2. Fixed infinite recursion bug in jde-find-project-file.
;;
;; Revision 1.11  1997/10/07 03:44:24  kinnucan
;; Required cl.
;;
;; Revision 1.10  1997/10/06 13:17:25  kinnucan
;; Removed last usage of obsolete bashify function.
;;
;; Revision 1.9  1997/10/06 04:02:27  kinnucan
;; Added jde-compiler variable and associated set command. Lets you
;; configure the JDE to use the compiler of your choice on a buffer
;; by buffer basis.
;;
;; Revision 1.8  1997/10/04 10:13:10  kinnucan
;; Added key bindings for menu commands.
;;
;; Revision 1.7  1997/10/03 05:57:54  kinnucan
;; 1. Revamped imenu regular expressions.
;; 2. Stopped quoting compile command arguments for bash under Win32.
;;
;; Revision 1.6  1997/10/01 03:13:04  kinnucan
;; Changed name of JDE menu from "Java" to "JDE" to avoid conflict
;; with cc-mode 5.18 menu, which is named "Java".
;;
;; Revision 1.5  1997/09/04 03:40:01  kinnucan
;; Updated version number.
;;
;; Revision 1.4  1997/09/04 03:38:13  kinnucan
;; 1. Made jde configuration variables buffer local to support automatic
;;    loading of project files.
;;
;; 2. Added Run Applet command to the jde menu.
;;
;; Revision 1.3  1997/08/28 02:54:17  kinnucan
;; Eliminated single quotes around path in jde-browse-jdk-doc.
;;
;; Revision 1.2  1997/08/26 08:50:29  kinnucan
;; Added jde-set-classpath command, which lets you set the classpath for
;; compiling and running applications.
;;
;; Revision 1.1  1997/06/18 17:25:57  paulk
;; Initial revision
;;
;; Revision 1.8  1997/06/18 17:20:00  paulk
;; Initial checkin.
;;

;;; Code:

(defvar jde-running-xemacs (string-match "XEmacs\\|Lucid" emacs-version))

(require 'jde-db)
(require 'jde-run)
(require 'compile)
(require 'imenu)
(require 'speedbar)
(require 'cl)

(if ( < emacs-major-version 20) 
    (require 'andersl-java-font-lock))

(defvar jde-web-browser  "netscape"
  "Browser for viewing Java doc")

(defvar jde-doc-dir "d:\\jdk1.1\\docs\\"
  "Path to directory containing the index.html for the JDK API doc.")

(defvar jde-compiler "javac"
"Pathname of the Java compiler to be used to compile the source
in this buffer. The default is javac.")

(defvar jde-compile-options ""
  "javac compile options, e.g., -depend -debug")

(defvar jde-classpath ""
  "Path to Java classes. The -classpath argument for the Java compiler
and virtual machine is set to the value of this variable.")

(defvar jde-project-file-name "prj.el"
  "Name of JDE project file. When it loads a Java source file, the
JDE looks for a lisp file of this name (the default is prj.el
in the source file hierarchy. If if finds such a file, it loads 
the file. You can use this file to set the classpath, compile options,
and other JDE options on a project-by-project basis.")

(defvar jde-hook nil
  "Hook called by Java IDE's java-mode-hook function.")

(defun jde-set-compiler (compiler)
  "Specify the pathname of the compiler to be used to compile the
current buffer. Default is javac."
  (interactive
   "sEnter compiler (javac): ")
   (if (string= compiler "")
       (setq jde-compiler "javac")
     (setq jde-compiler compiler)))


(defun jde-set-classpath (classpath)
  "Specify the value of the -classpath argument for the Java compiler and
interpreter."
  (interactive 
   "sEnter classpath: ")
  (setq jde-classpath classpath))

(defun jde-get-classpath-arg ()
  (if (and
       jde-classpath
       (> (length jde-classpath) 0))
      (concat "-classpath " jde-classpath)))


(defun jde-browse-jdk-doc ()
  "Displays JDK doc in a web browser. See jde-web-browser
and jde-doc-dir for more information."
  (interactive)
  (let ((cmd (concat jde-web-browser " "
		       jde-doc-dir "index.html")))
  (start-process-shell-command "jdkdoc" nil cmd)))

(defun jde-make-compile-command ()
  "Constructs the java compile command as: jde-compiler + options + buffer file name."
   (concat jde-compiler " " 
	   (if (jde-get-classpath-arg)
	       (concat (jde-get-classpath-arg) " "))
	   jde-compile-options " "
	   (file-name-nondirectory buffer-file-name)))

(defun jde-set-compile-options (options)
  "Sets the compile options.
Enter the options as you would on the command line, e.g.,
-depend -verbose."
  (interactive
   "sEnter options: ")
  (setq jde-compile-options options)
;;  (setq compile-command (jde-make-compile-command))
  )

(defun jde-compile (cmd)
  "Compile the Java program in the current buffer."
  (interactive "P")
  (setq compile-command (jde-make-compile-command))

  ;; Force save-some-buffers to use the minibuffer
  ;; to query user about whether to save modified buffers.
  ;; Otherwise, when user invokes jed-compile from
  ;; menu, save-some-buffers tries to popup a menu
  ;; which seems not to be supported--at least on
  ;; the PC.
  (if (eq system-type 'windows-nt)
      (let ((temp last-nonmenu-event))
	;; The next line makes emacs think that jde-compile
	;; was invoked from the minibuffer, even when it
	;; is actually invoked from the menu-bar.
	(setq last-nonmenu-event t)
	(save-some-buffers (not compilation-ask-about-save) nil)
	(setq last-nonmenu-event temp))
    (save-some-buffers (not compilation-ask-about-save) nil))

  (compile-internal compile-command "No more errors"))

(defvar jde-menu-map (make-sparse-keymap "JDE") nil)

(define-key jde-menu-map [jdk-doc] '("Browse JDK Doc" . jde-browse-jdk-doc))
(define-key jde-menu-map [speedbar] '("Speedbar" . speedbar-frame-mode))
(define-key jde-menu-map [run-applet] '("Run Applet" . jde-run-menu-run-applet))
(define-key jde-menu-map [debug-java] '("Debug App" . jde-db))
(define-key jde-menu-map [run-java] '("Run App" . jde-run))
(define-key jde-menu-map [compile-options]
	     '("Compile Options" . jde-set-compile-options))
(define-key jde-menu-map [compile] '("Compile" . jde-compile))

(defvar jde-xemacs-menu
  '(["Compile"           jde-compile t]
    ["Compile Options"   jde-set-compile-options t]
    ["Run App"           jde-run t]
    ["Debug App"         jde-db t]
    ["Run Applet"        jde-run-menu-run-applet t]
    ["Speedbar"          speedbar-frame-mode t]
    ["Browse JDK Doc"    jde-browse-jdk-doc t]
    )
  "XEmacs 19 menu for JDE.")

(defun jde-xemacs-menu ()
  (cons "JDE" jde-xemacs-menu))

(defun jde-set-menubar()
  (if (and (not (memq 'infodoc c-emacs-features))
	   (boundp 'current-menubar)
	   current-menubar)
      (if (fboundp 'add-submenu)
	  (add-submenu nil (jde-xemacs-menu))
	(add-menu nil "JDE" jde-xemacs-menu))
    (define-key java-mode-map [menu-bar java]
      (nconc (list "JDE") jde-menu-map)))    
)

(defun jde-java-mode-hook ()
  "Java IDE's java-mode-hook function.
This function sets  up the IDE menu and turns on font locking for Java syntax
highlighting. The last thing it does is call the jde-hook 
function, offering the user an opportunity to perform customizations."
	 (make-local-variable 'jde-web-browser)
	 (make-local-variable 'jde-doc-dir)
	 (make-local-variable 'jde-compile-options)
	 (make-local-variable 'jde-classpath)
	 (make-local-variable 'jde-compiler)
	 (make-local-variable 'jde-run-application-class)
	 (make-local-variable 'jde-run-java-vm)
	 (make-local-variable 'jde-run-java-vm-w)
	 (make-local-variable 'jde-run-args)
	 (make-local-variable 'jde-run-app-args)
	 (make-local-variable 'jde-run-applet-viewer)
	 (make-local-variable 'jde-run-applet-doc)
	 (make-local-variable 'jde-db-debugger)
	 (make-local-variable 'jde-db-debugger-is-executable)
	 (make-local-variable 'jde-db-args)
	 (make-local-variable 'jde-db-app-args)
	 (cond (window-system
		(define-key java-mode-map "\C-c\C-v\C-c" 'jde-compile)
		(define-key java-mode-map "\C-c\C-v\C-r" 'jde-run)
		(define-key java-mode-map "\C-c\C-v\C-d" 'jde-db)
		(define-key java-mode-map "\C-c\C-v\C-s" 'speedbar-frame-mode)
		(define-key java-mode-map "\C-c\C-v\C-a" 'jde-run-menu-run-applet)
		(define-key java-mode-map "\C-c\C-v\C-n" 'jde-browse-jdk-doc)
		(jde-set-menubar)

		(turn-on-font-lock)

		(setq font-lock-maximum-decoration t)

		(setq font-lock-support-mode 'lazy-lock-mode
		      lazy-lock-defer-on-scrolling t
		      lazy-lock-stealth-time 5
		      lazy-lock-stealth-verbose nil
		      )

		(setq case-fold-search nil)
		(setq imenu-generic-expression (jde-make-imenu-expression))
		(jde-load-project-file)
		(run-hooks 'jde-hook))))

(add-hook 'java-mode-hook 'jde-java-mode-hook)

;; Project File Functions

(defun jde-root-dir-p (dir)
  (let ((parent (concat dir "../")))
    (if (eq system-type 'windows-nt)
	(not (file-exists-p parent))
      (and 
       (string= (file-truename dir) "/")
       (string= (file-truename parent) "/")))))

(defun jde-find-project-file (dir)
  "Finds the project file for the Java source file in the current
buffer. Returns nil if it cannot find a project file in the
source file directory or an ascendant directory."
  (let ((file (find jde-project-file-name
		    (directory-files dir) :test 'string=)))
    (if file
	(concat dir file)
      (if (not (jde-root-dir-p dir))
	  (jde-find-project-file (concat dir "../"))))))

(defun jde-load-project-file ()
  "Loads the project file for the Java source file in the current
directory. Searches for the project file first in the source directory,
then in ascendant directories. Uses the first file that it encounters."
  (let ((prj-file (jde-find-project-file default-directory)))
    (if prj-file
	(load-file prj-file))))

(defun jde-open-project-file ()
  "Opens the project file for the Java source file in the
current buffer."
  (interactive)
  (let ((prj-file (jde-find-project-file default-directory)))
    (if prj-file
	(find-file prj-file)
      (message "%s" "Project file not found."))))



(defun jde-make-imenu-expression ()
  "Makes a replacement for the regular expression indexing  
patterns in imenu, which are too slow for the JDE's
speedbar. See `imenu-generic-expression'."
  (let* ((capital "A-Z\300-\326\330-\337")
	 (letter "a-zA-Z_$\300-\326\330-\366\370-\377")
	 (digit "0-9")
	 (white-space "\\s ")
	 (optional-white-spaces
	  (concat white-space "*"))
	 (bol "^")
	 (eol "^")
	 (anything ".*")

	 (primitive-type 
	  (concat "\\<\\(b\\(oolean\\|yte\\)"
		  "\\|char\\|double\\|float\\|int"
		  "\\|long\\|short\\|void\\)\\>"))
	 (primitive-type-count 2)

	 (primitive-type-no-void 
	  (concat "\\<\\(b\\(oolean\\|yte\\)"
		  "\\|char\\|double\\|float\\|int"
		  "\\|long\\|short\\)\\>"))
	 (primitive-type-no-void-count 2)

	 (identifier
	  (concat "\\<\\([" letter "][" letter digit "]*\\)\\>"))
	 (identifier-count 1)

	 ;; Class types are assumed to begin with a capital letter.
	 (class-type
	  (concat
	   "\\<\\([" capital "][" letter digit "]*\\)\\>"))
	 (class-type-count 1)

	 (modifier
	  (concat 
	   "\\<\\(abstract\\|const\\|final\\|native\\|"
	   "p\\(r\\(ivate\\|otected\\)\\|ublic\\)\\|"
	   "s\\(tatic\\|ynchronized\\)\\|transient\\|volatile\\)\\>"))
	 (modifier-count 4)

	 (optional-modifiers
	  (concat
	   "\\(" modifier optional-white-spaces "\\)*"))
	 (optional-modifiers-count 5)

	 (modifiers
	  (concat
	   "\\(" modifier optional-white-spaces "\\)+"))
	 (modifiers-count 5)

	 (optional-array-modifier
	  (concat
	   "\\(\\[" optional-white-spaces "\\]" optional-white-spaces "\\)*"))
	 (optional-array-modifier-count 1)

	 (class
	  (concat
	   bol
	   optional-white-spaces
	   optional-modifiers
	   "\\<class\\>"
	   optional-white-spaces
	   identifier))

	 (class-count (+ optional-modifiers-count
			  identifier-count))
	   
	 (interface
	  (concat
	   bol
	   optional-white-spaces
	   optional-modifiers
	   "\\<interface\\>"
	   optional-white-spaces
	   identifier))

	 (interface-count (+ optional-modifiers-count
			      identifier-count))

	 (constructor
	  (concat
	   bol
	   optional-white-spaces
	   modifiers                 ;; e.g., public
	   class-type                ;; e.g., Foo
	   optional-white-spaces
	   "("))

	 (constructor-count (+ optional-modifiers-count
			       class-type-count))

	 ;; Pattern for methods that return a primitive type
	 (method1
	  (concat
	   bol
	   anything
	   primitive-type          ;; e.g., int
	   optional-white-spaces
	   optional-array-modifier ;; e.g., []
	   identifier              ;; e.g., foo
	   optional-white-spaces
	   "("))

	 (method1-count (+ primitive-type-count
			   optional-array-modifier-count
			   identifier-count))
	
	 ;; Pattern for methods that return a class type
	 (method2
	  (concat
	   bol
	   anything
	   class-type
	   optional-white-spaces
	   optional-array-modifier
	   identifier
	   optional-white-spaces
	   "("))

	 (method2-count (+ class-type-count
			   optional-array-modifier-count
			   identifier-count))

	 (variable
	  (concat
	   bol
	   optional-white-spaces
	   optional-modifiers
	   optional-white-spaces
	   class-type
	   optional-white-spaces
	   optional-array-modifier
	   optional-white-spaces
	   identifier
	   optional-white-spaces
	   "\\(;\\|=\\)"))

	 (variable-count (+ optional-modifiers-count
			    class-type-count
			    optional-array-modifier-count
			    identifier-count))

	 (exp 
	  (`
	   (
	    (nil ;; methods index
	     (,  method1) (, method1-count))
	    (nil ;; methods index
	     (,  method2) (, method2-count))
	    ("Constructors" ;; methods index
	     (,  constructor) (, constructor-count))
	    ("Variables"
	     (, variable) (, variable-count))
	    ("Classes"
	     (, class) (, class-count))
	    ("Interfaces"
	     (, interface) (, interface-count))
	    )
	   ))
	 )
    exp))

(provide 'jde)

;;; jde.el ends here.







