;;Since the Elisp archive doesn't seem to work any more (mailing
;;contributions apparently has no effect) I am posting here the second
;;public release of a noweb mode for XEmacs.  Enjoy.
;;-------------------------------------------------------------------------

;;; noweb.el --- Mode for editing noweb source files

;; Copyright (C) 1997 Gerd Moellmann

;; Author:	  Gerd Moellmann <gerd@acm.org>
;; Maintainier:   Gerd Moellmann <gerd@acm.org>
;; Version:	  $Id: noweb.el,v 1.9 1997/05/21 18:40:06 gerd Exp $
;; Keywords:	  languages noweb

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; (This package has been used under XEmacs 19.13, 19.15.  FSF Emacs
;; is not supported.

;; This package provides a minor mode for XEmacs and FSF Emacs which
;; lets you edit a noweb buffer in language modes depending on whether
;; you are editing documentation or code parts of a web.  Which mode
;; to use is determined automatically by looking at chunk boundaries
;; and scrap names.  The package supports text highlighting with the
;; font-lock package.  Text and code parts of a web are highlighted
;; according to the mode in which they are edited.  Under XEmacs,
;; menus and marginal annotations provide navigation between chunks.
;; Spell checks can be performed on text parts only, ignoring code
;; chunks.

;; To install this package, compile it with M-x byte-compile-file and
;; copy the compiled file to a suitable location in your
;; `load-path'. Then add the following lines to your `.emacs':

;; (autoload 'noweb-mode "noweb" "Noweb mode" t)
;; (setq auto-mode-alist (append '("\\.nw$" . noweb-mode)
;; 				  auto-mode-alist))

;; Do _not_ add a call to noweb-mode to a major mode hook! Since
;; noweb-mode changes major modes, this will lead to infinite
;; recursion.

;;; Customization:

;; The mode in which text parts are edited is determined from the
;; value of the variable `noweb-text-mode'.  It defaults to
;; `latex-mode'.  The package works with AucTeX under XEmacs 19.13
;; which didn't come with AucTeX.

;; Code editing modes are determined in three ways.  If a scrap name
;; matches a pattern in `auto-mode-alist' the mode specified there is
;; used.  This means that a scrap named `file.el' will be edited in
;; Emacs lisp mode.  Scrap names can also contain mode strings similar
;; to those used as file variables: `-*-LANG-*-'.  If such a mode
;; string is found, the scrap is edited in `LANG-MODE'.  Otherwise the
;; code editing mode is `noweb-code-mode' which defaults to C mode.

;; Mode strings can be stripped in the noweb pipeline with a simple
;; filter like the following shell script:

;; #!/bin/sh
;; # nomode - strip language specifiers from code definitions
;; awk '	      
;; 	/^@defn/ {
;; 	    sub(/[ \t]*-\*-.*-\*-[ \t]*/, "", $0); 
;; 	}	      
;; 	{ print }'

;; The package expects to use `font-lock' highlighting. Text and code
;; parts are highlighted according to their mode as described above.
;; `fast-lock' and `lazy-lock' probably won't work with `noweb-mode'
;; because of the way major modes change over the buffer.  This hasn't
;; been tried, though.

;; When `noweb-use-annotations' is set to `t', marginal annotations
;; are added to a noweb buffer that let you jump to previous and next
;; uses of scrap names in the buffer.  Marginal annotations glyphs are
;; borrowed from `outl-mouse'.  They look like small up and down
;; arrows.  A small `d' or `u' indicates whether a definition or use
;; is in the indicated direction. Mouse-1 on an arrow jumps to the
;; previous or next scrap with the same name.  The function
;; `noweb-annotate-scraps' (also reachable via the Noweb menu) can be
;; called to re-build annotations.

;; `noweb-mode' adds a `Noweb' menu to the menu of the current major
;; mode popup menu and to the menu bar.  This menu contains some
;; navigational commands. Just take a look at it.

;;; History:

;; 1.3: Works with FSF Emacs 19.31, except for menus.
;; 1.2: Minor fixes
;; 1.1: Menus, annotations for XEmacs 19.13
;; 1.0: Posted to comp.programming.literate

;;; Todo:

;; - add a command to run a region of text through LaTeX (for testing
;;   XyPiC pictures.
;; - Commands over several web buffers, e.g., jumping to a definition
;;   over all loaded web buffers.
;; - install after-change-functon that makes sure yanked text is 
;;   highlighted properly.
;; - find a way to react immediately if options are customized.

;;; Known Bugs:

;; - infinite recursion when turning on font-lock in a major mode hook.
;; - a custimized option will take effect after a new buffer has been
;;   loaded only.

;; LCD Archive Entry:
;; noweb|Gerd Moellmann|gerd@acm.org|Minor mode for editing noweb webs|
;; 13-May-97|$Revision: 1.9 $| |

;;; Code:

(require 'cl)
(require 'font-lock)
(require 'advice)
(require 'custom)
(require 'annotations)
(require 'outl-mouse)			;for the nice arrow glyphs

(ad-start-advice)			;necessary?

;;; Switch off some warnings about unused variables.

(eval-when-compile
  (byte-compiler-options (warnings (- unused-vars))
			 (optimize t)))


;;; Customization

(defcustom noweb-fontify-chunks nil
  "Use of different fonts for text and code chunks in a web.

If this option is `on', `noweb-mode' will use different faces for 
code and text chunks. You can customize these fonts.  By default,
both text and code faces are equal so you won't find any difference
until you change the faces."
  :group 'noweb-fonts
  :type '(boolean)
  :tag "Different Fonts for Text and Code")

(defface noweb-custom-text-face '((t ()))
  "The face to use for text chunks.

Note that this face is only used when using different faces for 
code and text parts is switched on."
  :group 'noweb-fonts
  :tag "Text Chunk Face")

(defface noweb-custom-code-face '((t ()))
  "The face to use for code chunks.

Note that this face is only used when using different faces for 
code and text parts is switched on."
  :group 'noweb-fonts
  :tag "Code Chunk Face")

(defgroup noweb-fonts nil
  "Group for font usage."
  :tag "Fonts"
  :group 'noweb)


;;; Font-lock

(defface noweb-defn-face 
  '((((class color) (background dark))
     (:foreground "blue" :bold t))
    (((class color) (background light))
     (:foreground "darkblue" :bold t))
    (t
     (:bold t)))
  "Face used to highlight <<.*>>=."
  :tag "Face for scrap definitions"
  :group 'noweb-font-lock)

(defface noweb-use-face 
  '((((class color) (background dark))
     (:foreground "blue"))
    (((class color) (background light))
     (:foreground "darkblue"))
    (t 
     (:italic t :bold t)))
  "Face used to highlight <<.*>> uses."
  :tag "Face for Scrap Uses"
  :group 'noweb-font-lock)

(defface noweb-ref-face
  '((((class color) (background dark))
     (:foreground "blue"))
    (((class color) (background light))
     (:foreground "darkblue"))
    (t 
     (:itelic t)))
  "Face used to highlight [[.*]]."
  :tag "Face to hightlight references"
  :group 'noweb-font-lock)

(defgroup noweb-font-lock nil
  "Customization group for font-lock faces"
  :group 'noweb
  :tag "Hightlighting")


;;; Annotations stuff.

(defcustom noweb-use-annotations t
  "Building annotations on the left margin.

If this option is `on' `noweb-mode' will build annotations on the left
margin of web buffers that let you navigate between scraps."
  :group 'noweb-annotations
  :type '(boolean)
  :tag "Build Annotations")

(defcustom noweb-left-margin-width 8
  "The size of the left margin."
  :group 'noweb-annotations
  :type '(integer)
  :tag "Left Margin Width")

(defcustom noweb-left-margin-background-color "gray80"
  "The color to use for the left margin."
  :group 'noweb-annotations
  :type '(string)
  :tag "Left Margin Background")

(defgroup noweb-annotations nil
  "Customization group for annotations."
  :tag "Annotations"
  :group 'noweb)


;;; Keyboard

(defcustom noweb-prefix-key "\C-cn"
  "Prefix key for noweb mode commands."
  :type '(string)
  :tag "Prefix Key"
  :group 'noweb-keyboard)

(defcustom noweb-beware-electric-<-forms 
  '((not (eq major-mode 'c++-mode)))
  "List of forms to check if electric `<<...>>' insertion is desirable."
  :type '(sexp)
  :tag "Electric `<' Expansion"
  :group 'noweb-keyboard)

(make-variable-buffer-local 'noweb-beware-electric-<-forms)
(put 'noweb-beware-electric-<-forms 'permanent-local t)

(defgroup noweb-keyboard nil
  "Group for everything which has to do with the keyboard."
  :group 'noweb
  :tag "Keyboard")


;;; Default modes.

(defcustom noweb-text-mode 'latex-mode
  "The default mode to use for editing text parts of a web."
  :type '(symbol)
  :tag "Default Text Mode"
  :group 'noweb-modes)

(defcustom noweb-code-mode 'c-mode
  "The default mode for editing code parts of a web."
  :type '(symbol)
  :tag "Default Code Mode"
  :group 'noweb-modes)

(make-variable-buffer-local 'noweb-text-mode)
(put 'noweb-text-mode 'permanent-local t)
(make-variable-buffer-local 'noweb-code-mode)
(put 'noweb-code-mode 'permanent-local t)

(defgroup noweb-modes nil
  "Group for default noweb modes."
  :group 'noweb
  :tag "Modes")


(defgroup noweb nil
  "Group to customize noweb mode parameters."
  :group 'tools
  :group 'languages
  :tag "Noweb"
  :load 'noweb)


(defvar noweb-beware-forms '((not isearch-mode))
  "*List of forms to check to determine if mode changes are desirable.")

(defvar noweb-mode nil
  "Set to T if in noweb mode")
(make-variable-buffer-local 'noweb-mode)
(put 'noweb-mode 'permanent-local t)

(defvar noweb-mode-map nil
  "*The keymap used in noweb minor mode.")


(defvar noweb-mode-change-hook nil
  "Hooks run after major mode changes in noweb buffers.  Registered
functions are called with no arguments.")

(defvar noweb-scrap-name-start-regexp "<<"
  "*Regular expression matching the start of a scrap name. Definitions
are assumed to start in column 1 so that this regular expression,
with a prepended `^' is used to match the start of a definition.")

(defvar noweb-scrap-name-end-regexp ">>"
  "*Regular expression matching the end of a scrap name")

(defconst noweb-scrap-defn-name-end-regexp ">>="
  "*Regular expression matching the end of a scrap name in a 
definition.")

(defconst noweb-scrap-use-name-end-regexp ">>[^=]"
  "*Regular expression matching *only* the end of a scrap name in a 
scrap use.")

(defconst noweb-code-start-regexp 
  (concat "^" noweb-scrap-name-start-regexp "\\(.*\\)"
	  noweb-scrap-defn-name-end-regexp)
  "*Regular expression that matches the beginning of a code part.
Must include a sub-expression that matches the scrap name.")

(defconst noweb-text-start-regexp "^@"
  "*Regular expression matching the start of a text part.")

(defconst noweb-mode-string-regexp "-\\*-[ \t]*\\([^ \t]+\\)[ \t]*-\\*-"
  "A regular expression matching a mode string in a scrap name.")


;;; Use of different fonts in text and code parts (gimmik).

(defvar noweb-code-face (copy-face 'default 'noweb-code-face)
  "The face used for code chunks.")


;;; Local variables to preserve over major mode changes.

(defvar noweb-local-variables-to-make-permanent
  '(post-command-hook
    toolbar-ispell-function
    auto-show-mode 
    view-minor-mode view-exit-position view-exit-action view-prev-buffer)
  "*A list of local variables that should be preserved across major 
mode changes done by `noweb-mode'.

Local variables are preserved by giving them the `permanent-local'
property.  (I am not sure whether it is be a good idea to make
variables `permanent-local'. On one hand major mode changes are rare,
so probably a user will not notice any difference.  On the other hand
anything relying on `kill-all-local-variables'to really kill these
variables will break.)")


;;; Menu stuff

(defconst noweb-max-scrap-menu-items 30
  "The maximum number of menu items in a scrap menu. If more scraps
are found in a buffer, `noweb-mode' construct a tree menu structure.")

(defvar noweb-mode-menu
  '("Noweb" :included noweb-mode
    ("Scrap at Point" :included (noweb-in-scrap-name-p)
     ["Jump to First Occurrence" noweb-jump-to-first-occurrence :active t]
     ["Jump to Previous Occurrence" noweb-jump-to-previous-occurrence 
      :active (noweb-exists-previous-scrap-p)]
     ["Jump to Next Occurrence" noweb-jump-to-next-occurrence 
      :active (noweb-exists-next-scrap-p)]
     "--"
     ["List All Uses" noweb-list-all-uses :active t]
     ["List All Definitions" noweb-list-all-definitions :active t]
     ["List All" noweb-list-all-uses&definitions :active t])
    ("Goto First Scrap Definition" :filter noweb-goto-scrap-defn-filter)
    ("Goto First Scrap Use" :filter noweb-goto-scrap-use-filter)
    "--"
    ["Spell Text Parts" noweb-ispell-function :active t]
    ["Re-build Annotations" noweb-annotate-scraps :active t]
    ["Re-fontify" noweb-fontify-buffer :active t]
    "--"
    ["Use Different Fonts for Text and Code" noweb-toggle-fonts
     :active t :style toggle :selected noweb-fontify-chunks])
  "*The noweb menu definition.")


;;; Menu functions (XEmacs only).

(defun noweb-add-menu-to-menubar ()
  "Add noweb menu to the menu for the current major mode if the major
mode"
  (condition-case nil
      (find-menu-item current-menubar "Noweb" nil)
    (error (add-submenu nil noweb-mode-menu))))

(defun noweb-add-menu-to-popup-menu ()
  "Add the noweb menu to the buffers popup menu."
  (condition-case nil
      (when (and mode-popup-menu 
		 (null (car (find-menu-item mode-popup-menu
					    (list (car noweb-mode-menu))))))
	(setq mode-popup-menu (append mode-popup-menu
				      (list (vector "--" nil nil) 
					    noweb-mode-menu))))
    (error nil)))

(defun noweb-add-noweb-menu ()
  "Add the noweb menu to the menubar and to the current major mode
popup menu."
  (noweb-add-menu-to-menubar)
  (noweb-add-menu-to-popup-menu))


;;; Noweb mode.

(defun noweb-mode (&optional arg)
  "Turn on/off noweb mode. `noweb-mode' will let you edit text and
code chunks of your web in suitable major modes with `font-lock'
highlighting depending on the modes used.

The mode for text parts is determined from the buffer local variable
`noweb-text-mode' which is `latex-mode' by default.  The mode to use
for code chunks is determined in three ways: If the scrap name of the
chunk matches a regular expression in `auto-mode-alist', the mode
there will be used.  If the scrap name contains a string of the form
`-*-LANGUAGE-*-' the chunk will be edited in `LANGUAGE-mode'.  If both
is not the case, `noweb-code-mode' is used which is a buffer local
variable defaulting to `c-mode'.

\\{noweb-mode-map}"
  (interactive "P")
  (setq noweb-mode (if (null arg) 
		       (not noweb-mode)
		     (> (prefix-numeric-value arg) 0)))
  (if noweb-mode
      (progn
	;; Make sure `font-lock-fontified' is local, otherwise other
	;; buffers won't be fontified when it is set inadvertantly.
	(loop for sym in noweb-local-variables-to-make-permanent do
	      (put sym 'permanent-local t))
	(make-local-variable 'font-lock-fontified)
	(set (make-local-variable 'revert-buffer-function) 'noweb-revert-buffer)
	(set (make-local-variable 'toolbar-ispell-function) 'noweb-ispell-function)
	(noweb-fontify-buffer)
	(make-local-hook 'post-command-hook)
	(add-hook 'post-command-hook 'noweb-post-command-hook-fn)
	(run-hooks 'noweb-mode-hook))
    (remove-hook 'post-command-hook 'noweb-post-command-hook-fn)
    (setq revert-buffer-function nil))
  (redraw-modeline))


;;; Switching modes.

(defsubst noweb-mode-from-auto-mode-alist (scrap-name)
  "Scan `auto-mode-alist' for a mode matching `scrap-name'. Return
the match found or nil."
  (rest (find-if #'(lambda (x) (string-match (first x) scrap-name))
                 auto-mode-alist)))

(defsubst noweb-mode-from-mode-string (scrap-name)
  "Return a mode from a mode string in `scrap-name'.  Mode strings have
the usual form described by `noweb-mode-string-regexp'.  Returns nil if
the scrap name does not contain a mode string."
  (let (mode)
    (when (string-match noweb-mode-string-regexp scrap-name)
      (let ((language 
	     (downcase (substring scrap-name (match-beginning 1) (match-end 1)))))
	(setf mode (intern-soft (concat language "-mode")))
	(when (and mode (not (fboundp mode)))
	  (setf mode nil))))
    mode))

(defsubst noweb-code-mode (scrap-name)
  "Determine the mode for a code chunk from a given scrap name
`scrap-name'.  If `scrap-name' matches a pattern in `auto-mode-alist',
use the mode from there.  If `scrap-name' contains a mode string of
the form \"-*-LANGUAGE-*-\" use `LANGUAGE-mode'.  Otherwise, return
the default code mode `noweb-code-mode'."
  (or (noweb-mode-from-auto-mode-alist scrap-name)
      (noweb-mode-from-mode-string scrap-name)
      noweb-code-mode))

(defun noweb-switch-to-mode (mode &optional leave-menus-alone-p)
  "Switch to major mode MODE."
  (let ((switch-p (not (eq major-mode mode))))
    (when switch-p
      (let ((font-lock-fontified t))
	;; Reset the menubar of this buffer because we do not want the C
	;; menu in LaTeX parts, for example.  This has the unfortunate
	;; effect of removing menus that are added by packages like
	;; `function-menu' on a per mode basis, but there is nothing we
	;; can do about it.
 	(if leave-menus-alone-p
	    (funcall mode)
	  (set-buffer-menubar default-menubar)
	  (setq mode-popup-menu default-popup-menu)
	  (funcall mode)
	  (noweb-add-noweb-menu)
	  (noweb-ensure-left-margin))
	(run-hooks 'noweb-mode-change-hook)
	(make-local-hook 'post-command-hook)
	(add-hook 'post-command-hook 'noweb-post-command-hook-fn)
	(turn-on-font-lock)))
    switch-p))


;;; Post command hook.

(defun noweb-post-command-hook-fn ()
  "Called after every command in a buffer. If we are in noweb mode
and there is nothing to prevent us from switching modes (see
NOWEB-BEWARE-FORMS), try if we have to change modes and do so."
  (when (and noweb-mode (eval (cons 'and noweb-beware-forms)))
    (let* ((scrap-end 
	    (save-excursion (re-search-backward noweb-text-start-regexp nil t))) 
	   (scrap-start
	    (save-excursion (re-search-backward noweb-code-start-regexp nil t)))
	   (mode (if (or (and (null scrap-start) (null scrap-end))
			 (and scrap-end (< scrap-start scrap-end)))
		     noweb-text-mode
		   (noweb-code-mode (buffer-substring (match-beginning 1) 
						      (match-end 1))))))
	(noweb-switch-to-mode mode))))


;;; Fontification.

(defmacro noweb-for-all-text-parts (start-point &rest forms)
  "Evaluate form on all text parts.  START and END will be bound
to the start and end of the region of a text part, when FORM is
evaluated."
  `(let (start end)
     (goto-char ,start-point)
     (setq start (point))
     (noweb-switch-to-mode noweb-text-mode 'leave-menus-alone)
     (while (setq end (re-search-forward noweb-code-start-regexp nil t))
       ,@forms
       (re-search-forward noweb-text-start-regexp nil t)
       (setq start (point)))
     (when (< start (point-max))
       (setq end (point-max))
       ,@forms)))

(defmacro noweb-for-all-code-parts (start-point &rest forms)
  "Evaluate form on all code parts.  START and END will be bound
to the start and end of the region of a text part, when FORM is
evaluated."
  `(let (start)
     (goto-char ,start-point)
     (while (setq start (re-search-forward noweb-code-start-regexp nil t))
       (let ((scrap (buffer-substring (match-beginning 1) (match-end 1)))
	     (end (re-search-forward noweb-text-start-regexp nil t)))
	 (when end
	   (noweb-switch-to-mode (noweb-code-mode scrap) 'leave-menus-alone)
	   ,@forms)))))

(put 'noweb-for-all-text-parts 'lisp-indent-function 1)
(put 'noweb-for-all-code-parts 'lisp-indent-function 1)

(defmacro noweb-fontify-region (start end)
  "Do the necessary steps to fontify a region `start' and `end'."
  (if (fboundp 'font-lock-hack-keywords)
      `(save-excursion
	 (font-lock-fontify-region ,start ,end)
	 (font-lock-hack-keywords ,start ,end))
    `(save-excursion
       (font-lock-fontify-region ,start ,end))))

(defsubst noweb-make-code-extent (start end)
  (unless (extent-at start (current-buffer) 'noweb)
    (let ((extent (make-extent start end)))
      (set-extent-property extent 'noweb 'code-part)
      (set-extent-face extent 'noweb-code-face))))

(defun noweb-unfontify-buffer ()
  "Remove fontification of the current buffer."
  (font-lock-unfontify-region (point-min) (point-max))
  (setq font-lock-fontified nil))

(defun noweb-fontify-buffer ()
  "Fontify a whole noweb buffer. Text and code parts are fontified
according to the mode they are edited in."
  (interactive)
  (save-restriction
    (widen)
    (let ((initial-mode major-mode))
      (condition-case nil
	  (save-excursion
	    (save-match-data
	      (noweb-unfontify-buffer)
	      (message "Fontifying text parts...")
	      (noweb-for-all-text-parts (point-min) 
		(noweb-fontify-region start end))
	      (message "Fontifying code parts...")
	      (noweb-for-all-code-parts (point-min)
		(noweb-make-code-extent start end)
		(noweb-fontify-region start end))
	      (noweb-switch-to-mode initial-mode)
	      (setq font-lock-fontified t))
	    (when noweb-use-annotations
	      (message "Adding annotations...")
	      (noweb-annotate-scraps))
	    (message ""))
	(quit 
	 (noweb-switch-to-mode initial-mode)
	 (noweb-unfontify-buffer))))))


;;; Commands.

(defun noweb-scrap-name-alist (&optional exclude-start exclude-end)
  "Build and return a ALIST of all scrap definitions and uses in the 
current buffer, except those between EXCLUDE-START and EXCLUDE-END.
The CDR of each cons cell in the ALIST is another CONS with CAR
equal to point where the scrap was found and CDR being T if it is a
definition."
  (let (alist)
    (save-excursion
      (save-restriction
        (widen)
        (goto-char (point-min))
        (while (re-search-forward "[^@]<<\\([^\r\n]*\\)>>" nil t)
          (when (or (or (null exclude-start) (< (point) exclude-start))
                    (or (null exclude-end) (> (point) exclude-end)))
	    (let ((scrap (buffer-substring (match-beginning 1)
					   (match-end 1))))
	      (push (cons (noweb-strip-scrap-name scrap)
			  (cons (point) (looking-at "=")))
		  alist))))))
    (nreverse alist)))

(defun noweb-scrap-name-set (&optional exclude-start exclude-end)
  "Build and return a ALIST of all scraps in the current buffer.
Don't mention a scrap name more than once."
  (remove-duplicates (noweb-scrap-name-alist exclude-start exclude-end)
		     :key 'car :test 'string=))

(defun noweb-list-to-forest (list nitems)
  "Given a LIST of items, transform it into a list of trees so 
that the result has no more than NITEMS items."
  (if (> (length list) nitems)
      (let (sublist i result)
	(while list
	  (setf i 0 sublist nil)
	  (while (and list (< i nitems))
	    (incf i)
	    (setq sublist (cons (car list) sublist))
	    (setq list (cdr list)))
	  (setf result (cons (nreverse sublist) result)))
	(noweb-list-to-forest (nreverse result) nitems))
    list))

(defun noweb-insert-menu-titles (forest)
  "Given a list of trees as built by `noweb-list-to-forest', insert
menu title strings at the start of each non-leaf node."
  (let ((list forest))
    (while list
      (when (listp (car list))
	(setf (car list) (noweb-insert-menu-titles (car list))))
      (setq list (cdr list)))
    (if (listp (car forest))
	(cons (car (car forest)) forest)
      (cons (concat (aref (car forest) 0) "...") forest))))

(defun noweb-scrap-name-items (callback)
  "Build a menu structure from all scrap names in the current buffer.
CALLBACK is a function to be called when a menu item is activated.
It will get passed the title of the menu item when called."
  (flet ((make-item (x) (vector (car x) (list callback (car x)) t))
	 (sort-items (x y) (string< (aref x 0) (aref y 0))))
    (let ((scrap-list (mapcar 'make-item (noweb-scrap-name-set))))
      (setq scrap-list (sort scrap-list 'sort-items))
      (setq scrap-list (noweb-list-to-forest scrap-list noweb-max-scrap-menu-items))
      ;; The title of the top-most menu structure is not used
      (cdr (noweb-insert-menu-titles scrap-list)))))

(defun noweb-goto-scrap-defn-filter (list-of-items)
  "Return a menu structure in which each menu item activation calls
NOWEB-GOTO-SCRAP with the name of the menu item (the scrap name)."
  (noweb-scrap-name-items 'noweb-goto-scrap-defn))

(defun noweb-goto-scrap-use-filter (list-of-items)
  "Return a menu structure in which each menu item activation calls
NOWEB-GOTO-SCRAP-USE with the name of the menu item (the scrap name)."
  (noweb-scrap-name-items 'noweb-goto-scrap-use))

(defun noweb-goto-scrap (scrap what)
  "Jump to a definition or use of `scrap'. `what' is either 'definition
or 'use."
  (let ((regexp (noweb-build-scrap-regexp scrap what)))
    (goto-char (point-min))
    (unless (re-search-forward regexp nil t)
      (error "Not found."))
    (goto-char (match-beginning 1))
    (noweb-recenter)))

(defun noweb-goto-scrap-defn (scrap)
  "Menu action that jump to the first definition of `scrap'."
  (noweb-goto-scrap scrap 'definition))

(defun noweb-goto-scrap-use (scrap)
  "Menu action that jump to the first use of `scrap'."
  (noweb-goto-scrap scrap 'use))


;;; Electric text insertion

(defun noweb-insert-noweb-scrap-name-in-code-chunk ()
  "Insert a <<>> at point if appropriate. Return T if inserted."
  (let (inserted)
    (when (eval (cons 'and noweb-beware-electric-<-forms))
      (cond ((bolp)
	     (insert "<<>>")
	     (backward-char 2)
	     (indent-according-to-mode)
	     (setq inserted t))
	    ((and (char-equal (preceding-char) ?\<)
		  (not (char-equal (char-after (- (point) 2)) ?\@)))
	     (insert "<>>")
	     (backward-char 2)
	     (indent-according-to-mode)
	     (setq inserted t))))
    inserted))

(defun noweb-insert-noweb-scrap-name-in-text-chunk ()
  "Insert text for a noweb scrap definition into buffer if appropriate.
Return T if inserted."
  (let (inserted)
    (when (bolp)
      (insert "<<>>=\n")
      (save-excursion (insert "\n@ %def \n"))
      (noweb-make-code-extent (point) (+ 1 (point)))
      (backward-char 4)
      (setq inserted t))
    inserted))

(defun noweb-electric-< ()
  "Insert scrap uses or definitions into the buffer depending
on where we are in the web."
  (interactive)
  (unless (or (and (looking-at "[ \t]*$")
		   (noweb-in-code-chunk-p)
		   (noweb-insert-noweb-scrap-name-in-code-chunk))
	      (noweb-insert-noweb-scrap-name-in-text-chunk))
    (self-insert-command 1)))

(defun noweb-make-region-reference (arg)
  "Make selected region a reference. With a prefix arg, also
surround the reference with `...'"
  (interactive "P")
  (unless (region-active-p) 
    (error "The mark is not active now"))
  (save-excursion
    (let ((prefix (if arg "`[[" "[["))
	  (suffix (if arg "]]'" "]]")))
      (goto-char (region-end))
      (insert suffix)
      (goto-char (region-beginning))
      (insert prefix))))

(defun noweb-electric-bracket (&optional arg)
  "Insert a noweb reference.  If characters follow point, tries
to be smart and inserts the reference around the next word, unless
given a prefix key."
  (interactive "P")
  (when (and (> (point) (point-min))
	     (char-equal (char-after (1- (point))) ?\[))
    (save-excursion
      (unless (or arg (looking-at "[ \t]*$"))
	(skip-chars-forward "a-zA-Z0-9_"))
      (insert "]]")))
  (self-insert-command 1))

(defsubst noweb-start-of-line-point ()
  "Return point at the start of the line."
  (save-excursion (beginning-of-line) (point)))

(defsubst noweb-end-of-line-point ()
  "Return point at the end of the line."
  (save-excursion (end-of-line) (point)))

(defun noweb-scrap-name-start ()
  "Return point at which a scrap name starts.  Returns NIL if not 
in a scrap name."
  (save-excursion
    (and (re-search-backward noweb-scrap-name-start-regexp 
			     (noweb-start-of-line-point) t)
	 (match-end 0))))

(defun noweb-scrap-name-end ()
  "Return the point at which a scrap name ends.  Returns NIL if 
the end of the scrap could not be found (on the same line)."
  (save-excursion
    (and (re-search-forward noweb-scrap-name-end-regexp 
			    (noweb-end-of-line-point) t)
	 (match-beginning 0))))

(defun noweb-in-scrap-name-p ()
  "Return T if inside a scrap name."
  (and (not (null (noweb-scrap-name-start)))
       (not (null (noweb-scrap-name-end)))))

(defun noweb-push-completing-events ()
  "Depending on the Emacs version, push events so that a we will see
the maximal completion in the minibuffer without hitting TAB manually."
  (push (character-to-event ?\t) unread-command-events)
  (push (character-to-event ?\t) unread-command-events))

(defun noweb-complete-scrap ()
  "Complete the scrap name in front of point from the scraps used
in the current buffer."
  (interactive)
  (if (noweb-in-scrap-name-p)
      (let* ((start (noweb-scrap-name-start))
	     (scrap (buffer-substring start (point)))
	     (line-start (noweb-start-of-line-point))
	     (line-end (noweb-end-of-line-point))
	     (alist (noweb-scrap-name-set line-start line-end))
	     (completion-ignore-case t)
	     (choice (progn
		       (noweb-push-completing-events)
		       (completing-read "Scrap name: " alist nil nil scrap))))
	(when choice
	  (delete-region start (or (noweb-scrap-name-end) line-end))
	  (insert choice)))
    (indent-according-to-mode)))

(defun noweb-trim (string what)
  "Strip white space from STRING.  Second argument WHAT is a symbol
specifying from which end(s) of the string white space should be stripped.
It can be 'both, 'leading or 'trailing."
  (when (and (memq what '(both leading))
	     (string-match "^[ \t]+" string))
    (setq string (substring string (match-end 0))))
  (when (and (memq what '(both trailing))
	     (string-match "[ \t]+$" string))
    (setq string (substring string 0 (match-beginning 0))))
  string)

(defun noweb-strip-scrap-name (scrap)
  "Strip mode strings, ellipsis and white space from SCRAP and return 
the result."
  (let ((mode-string (string-match noweb-mode-string-regexp scrap))
	(ellipsis (string-match "\\.\\.\\." scrap)))
    (when (or mode-string ellipsis)
      (let* ((length (length scrap))
	     (end (min (or mode-string length) (or ellipsis length))))
	(setq scrap (substring scrap 0 end))))
    (noweb-trim scrap 'both)))

(defun noweb-extract-scrap-name-from-point ()
  "Get the scrap name around point. Strip mode string and ellipsis
if present (mode string must follow ellipsis)."
  (let ((start (or (noweb-scrap-name-start) (error "No scrap name start")))
	(end (or (noweb-scrap-name-end) (error "No scrap name end"))))
    (noweb-strip-scrap-name (buffer-substring start end))))

(defun noweb-build-scrap-regexp (scrap-name what)
  "Build a regular expression for matching uses, definitions
or both of scrap SCRAP-NAME.  WHAT is a symbol.  It is `definition'
if definitions should be searched, `use' if uses should be searched,
and `definition-and-use' when both should be searched."
  (setq scrap-name (concat "\\(" (regexp-quote scrap-name) ".*\\)"))
  (case what
    (use 
     (concat noweb-scrap-name-start-regexp scrap-name 
	     noweb-scrap-use-name-end-regexp))
    (definition 
      (concat "^" noweb-scrap-name-start-regexp scrap-name 
	      noweb-scrap-defn-name-end-regexp))
    (t (concat noweb-scrap-name-start-regexp scrap-name 
	       noweb-scrap-name-end-regexp))))


;;; Jumping to scraps.

(defun noweb-recenter ()
  "Recenter without redrawing the window."
  (let ((prefix-arg t))
    (recenter (/ (window-height) 2))))

(defun noweb-jump (direction &optional start-at-bobp)
  "Search in `direction' for the next or previous occurrence of 
the scrap name at point.  If `start-at-bobp' is not nil, begin the
search at `point-min'."
  (unless (noweb-in-scrap-name-p)
    (error "Not in a scrap name"))
  (let* ((scrap (noweb-extract-scrap-name-from-point))
	 (regexp (noweb-build-scrap-regexp scrap 'definition-and-use))
	 (search-fn (if (eq direction 'forward) 
			're-search-forward 're-search-backward)))
    (if (eq direction 'forward)
	(if start-at-bobp 
	    (goto-char (point-min))
	  (end-of-line))
      (beginning-of-line))
    (unless (funcall search-fn regexp nil t)
      (error "No match found"))
    (goto-char (match-beginning 1))
    (noweb-recenter)))

(defun noweb-jump-to-first-occurrence ()
  "Jump to the first occurrence of scrap at point in the current buffer."
  (interactive)
  (noweb-jump 'forward 'start-at-bobp))

(defun noweb-jump-to-next-occurrence (&optional start-at-bobp)
  "Jump to the next occurrence of scrap at point in current buffer."
  (interactive)
  (noweb-jump 'forward))

(defun noweb-jump-to-previous-occurrence ()
  "Jump to previous occurrence of scrap at point in current buffer."
  (interactive)
  (noweb-jump 'backward))


;;; Menu activation functions.

(defun noweb-in-code-chunk-p ()
  "Return T if point is in a code chunk."
  (save-excursion
    (beginning-of-line)
    (while (and (not (bobp))
                (not (looking-at noweb-text-start-regexp))
                (not (looking-at noweb-code-start-regexp)))
      (previous-line 1)
      (beginning-of-line))
    (looking-at noweb-code-start-regexp)))

(defun noweb-exists-previous-scrap-p ()
  "Return t if a previous use of scrap at point exists."
  (let ((scrap (noweb-extract-scrap-name-from-point)))
    (save-excursion
      (beginning-of-line)
      (re-search-backward (noweb-build-scrap-regexp scrap 'definition-and-use)
			  nil t))))
                     
(defun noweb-exists-next-scrap-p ()
  "Return t if a next use of scrap at point exists."
  (let ((scrap (noweb-extract-scrap-name-from-point)))
    (save-excursion
      (end-of-line)
      (re-search-forward (noweb-build-scrap-regexp scrap 'definition-and-use)
			 nil t))))

                     
;;; Listing occurrences of a scrap name

(defun noweb-list-all-definitions ()
  "Pop up an `occur' buffer that lists all definitions of the scrap
at point."
  (interactive)
  (let ((scrap (noweb-extract-scrap-name-from-point)))
    (list-matching-lines (noweb-build-scrap-regexp scrap 'definition))))

(defun noweb-list-all-uses ()
  "Pop up an `occur' buffer that lists all uses of the scrap at point."
  (interactive)
  (let ((scrap (noweb-extract-scrap-name-from-point)))
    (list-matching-lines (noweb-build-scrap-regexp scrap 'use))))

(defun noweb-list-all-uses&definitions ()
  "Pop up an `occur' buffer that lists all uses and definitions of the 
scrap at point."
  (interactive)
  (let ((scrap (noweb-extract-scrap-name-from-point)))
    (list-matching-lines (noweb-build-scrap-regexp scrap 'definition-and-use))))


(defadvice font-lock-fontify-buffer 
  (around noweb-font-lock-buffer first freeze)
  (if noweb-mode
      (noweb-fontify-buffer)
    ad-do-it))

(defun noweb-revert-buffer (ignore-auto no-confirm)
  "Function used as REVERT-BUFFER-FUNCTION in noweb mode. This
solves the problem that minor modes are lost when reverting 
webs (which happens automatically when pcl-cvs is used."
  (setq revert-buffer-function nil)
  (revert-buffer ignore-auto no-confirm)
  (noweb-mode))


;;; Spell checking

(defun noweb-ispell-function ()
  "Call `ispell-region' on either the active region, if any, or on all
text parts of this buffer beginning with the text part point is in or,
if we are in a code part, the text part in front of it.  Spell check
ends when the ispell process is killed (ispells `q' command).

Note that the XEmacs toolbar icon invokes this function in noweb
buffers.

This function `misuses' the variables `ispell-skip-tib', 
`ispell-tib-ref-beginning' and `ispell-tib-ref-end' to let ispell
skip over `[[...]]' references in text parts." 
  (interactive)
  (let ((ispell-skip-tib t)
	(ispell-tib-ref-beginning "\\[\\[")
	(ispell-tib-ref-end "]]"))
    (if (region-active-p)
	(ispell-region (region-beginning) (region-end))
      (let ((text-start (or (re-search-backward noweb-text-start-regexp nil t)
			    (point-min))))
	(block 'spell-check
	  (noweb-for-all-text-parts text-start
	    (ispell-region start end)
	    (when (null (symbol-value 'ispell-process))
	      (return-from 'spell-check))))))))


;;; Annotations. These are only available under XEmacs.

(defun noweb-search (regexp search-fn)
  (let ((found (save-excursion (funcall search-fn regexp nil t))))
    (when (and (not found)
	       (y-or-n-p "Nothing found. Re-build annotations? "))
      (noweb-annotate-scraps)
      (setq found (save-excursion (funcall search-fn regexp nil t))))
    (unless found
      (error "Nothing found."))
    (goto-char (match-beginning 1))
    (noweb-recenter)))

(defun noweb-search-forward (regexp event)
  "Function called when a down arrow annotation is clicked."
  (goto-char (window-start))
  (next-line (event-y event))
  (end-of-line)
  (noweb-search regexp 're-search-forward))

(defun noweb-search-backward (regexp event)
  "Function called when an up arrow annotation is clicked."
  (goto-char (window-start))
  (next-line (event-y event))
  (noweb-search regexp 're-search-backward))

(defun noweb-add-annotation (pos glyph mask regexp function)
  (let ((annotation (make-annotation glyph pos 'outside-margin
				     nil t mask)))
    (set-annotation-data annotation regexp)
    (set-annotation-action annotation function)))

(defun noweb-add-annotations-to-scrap (scrap prev next)
  "Add marginal annotations to all scrap definitions and uses in
the current buffer."
  (let ((pos (cadr scrap)))
    (when prev
      (let ((glyph outline-up-arrow)
	    (mask outline-up-arrow-mask)
	    (regexp (noweb-build-scrap-regexp (car scrap) 'definition-or-use)))
	(noweb-add-annotation pos glyph mask regexp 'noweb-search-backward)
	(noweb-add-annotation pos (if (cddr prev) "d " "u ") nil regexp 
			      'noweb-search-backward)))
    (when next
      (let ((glyph outline-down-arrow)
	    (mask outline-down-arrow-mask)
	    (regexp (noweb-build-scrap-regexp (car scrap) 'definition-or-use)))
	(noweb-add-annotation pos glyph mask regexp 'noweb-search-forward)
	(noweb-add-annotation pos (if (cddr next) "d " "u ") nil regexp 
			      'noweb-search-forward)))))
					
(defun noweb-ensure-left-margin ()
  "Make sure that the left margin width is at least NOWEB-LEFT-MARGIN-WIDTH
wide."
  (let* ((buffer (current-buffer))
	 (left-margin (specifier-specs left-margin-width buffer)))
    (when (or (null left-margin)
	      (< left-margin noweb-left-margin-width))
      (add-spec-to-specifier left-margin-width 
			     noweb-left-margin-width
			     buffer))))
  
(defun noweb-ensure-margin-face ()
  "Set the left margin background."
  (set-face-background 'left-margin noweb-left-margin-background-color))

(defun noweb-annotate-scraps ()
  "For each scrap in the current buffer, insert annotations that let
you jump to the next and previous uses and definitions of the same
scrap."
  (interactive)
  (noweb-ensure-left-margin)
  (noweb-ensure-margin-face)
  (mapc 'delete-annotation (annotation-list))
  (flet ((name/point< (x y) (if (string= (car x) (car y))
				(< (cadr x) (cadr y))
			      (string< (car x) (car y))))
	 (scrap= (x y) (string= (car x) (car y))))
    (let ((scraps (sort (noweb-scrap-name-alist) 'name/point<)))
      (while scraps
	(let ((seq (car scraps)) prev)
	  (while (and scraps (scrap= (car scraps) seq))
	    (let ((current (car scraps))
		  (next (and (scrap= (cadr scraps) seq) 
			     (cadr scraps))))
	      (noweb-add-annotations-to-scrap current prev next)
	      (setq scraps (cdr scraps) prev current))))))))


;;; Use of different fonts for code and text parts.

(defun noweb-copy-face-properties (from-face from-locale to-face
							  to-locale)
  (let ((font (face-font from-face from-locale)))
    (when font
      (set-face-font to-face font to-locale))))

(defun noweb-toggle-fonts (&optional force-on)
  "Toggle use of different faces for code and text chunks."
  (interactive)
  (setq noweb-fontify-chunks (or force-on (not noweb-fontify-chunks)))
  (let ((buffer (current-buffer)))
    (if noweb-fontify-chunks
	(progn
	  (noweb-copy-face-properties 'noweb-custom-text-face 'global
				      'default buffer)
	  (noweb-copy-face-properties 'noweb-custom-code-face 'global
				      'noweb-code-face buffer))
      (noweb-copy-face-properties 'default 'global 'default buffer)
      (noweb-copy-face-properties 'default 'global 'noweb-code-face buffer))))

(defun noweb-re-fontify-code-part ()
  (interactive)
  (unless (noweb-in-code-chunk-p)
    (error "Not in a code chunk."))
  (let ((start (save-excursion 
		 (re-search-backward noweb-code-start-regexp nil t)
		 (1+ (match-end 0))))
	(end (or (save-excursion 
		   (re-search-forward noweb-text-start-regexp nil t))
		 (point-max))))
    (font-lock-unfontify-region start end)
    (noweb-fontify-region start end)))


;;; Commands over all noweb buffers


;;; Narrowing

(defun noweb-narrow-to-code-part ()
  "Narrow buffer to current code part."
  (interactive)
  (unless (noweb-in-code-chunk-p)
    (error "Not in a code chunk."))
  (let ((start (save-excursion 
		 (re-search-backward noweb-code-start-regexp nil t)
		 (1+ (match-end 0))))
	(end (or (save-excursion 
		   (re-search-forward noweb-text-start-regexp nil t))
		 (point-max))))
    (narrow-to-region start (1- end))))


;;; Things done when this file is loaded.
  
(unless noweb-mode-map
  (setq noweb-mode-map (make-sparse-keymap))
  (define-key noweb-mode-map "<" 'noweb-electric-<)
  (define-key noweb-mode-map "[" 'noweb-electric-bracket)
  (define-key noweb-mode-map "\t" 'noweb-complete-scrap)
  (let ((map (make-sparse-keymap)))
    (define-key map "f" 'noweb-re-fontify-code-part)
    (define-key map "n" 'noweb-jump-to-next-occurrence)
    (define-key map "r" 'noweb-make-region-reference)
    (define-key map "c" 'noweb-narrow-to-code-part)
    (define-key map "p" 'noweb-jump-to-previous-occurrence)
    (define-key noweb-mode-map noweb-prefix-key map)))

(defun noweb-install-font-lock-keywords ()
  "In FSF Emacs 19.31 font-lock, defaults are stored in a let binding
in `font-lock- defaults-alist'.  This means that we have to do
some kludge."
  (let ((references '("\\[\\[[^]]*\\]\\]" 0 noweb-ref-face t))
	(uses '("<<.*>>" 0 noweb-use-face t))
	(definitions '("^<<.*>>=" 0 noweb-defn-face t))
	(c '(c-font-lock-keywords 
	     c-font-lock-keywords-1 c-font-lock-keywords-2
	     c-font-lock-keywords-3 c++-font-lock-keywords 
	     c++-font-lock-keywords-1 c++-font-lock-keywords-2 
	     c++-font-lock-keywords-3))
	(tex '(tex-font-lock-keywords
	       font-latex-keywords font-latex-keywords-1
	       font-latex-keywords-2)))
    (mapc #'(lambda (x) 
	      (when (boundp x) 
		(set x (cons references (symbol-value x)))
		(set x (cons definitions (symbol-value x)))))
	  tex)
    (mapc #'(lambda (x) 
	      (when (boundp x) 
		(set x (cons uses (symbol-value x)))))
	  c)))

;; Changes between using fonts and colors are done by setting variables,
;; then calling `font-lock-mode'.  This hook makes sure that we use the
;; right faces for noweb.
(defun noweb-font-lock-mode-hook-fn ()
  (when (and noweb-mode noweb-fontify-chunks)
    (noweb-toggle-fonts 'on)))

(add-hook 'font-lock-mode-hook 'noweb-font-lock-mode-hook-fn)

(unless (assq 'noweb-mode minor-mode-alist)
  (push '(noweb-mode " Noweb") minor-mode-alist)
  (push (cons 'noweb-mode noweb-mode-map) minor-mode-map-alist)
  (noweb-install-font-lock-keywords))


(provide 'noweb)

;;; noweb.el ends here


