;; protogen-mode.el --- sh (protogen, bash) script editing mode for GNU Emacs.

;; -- Copied from ksh-mode.el 

;; Copyright (C) 1992-95 Gary Ellison.

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;; LCD Archive Entry:
;; protogen-mode|Gary F. Ellison|Gary_F_Ellison@ATT.COM
;; Mode for editing sh/protogen/bash scripts
;; 23-Feb-95|2.6|~/modes/ksh-mode.el.Z|

;; Author: Gary F. Ellison <Gary.F.Ellison@ATT.COM>
;;                   AT&T Bell Laboratories
;;                   6200 East Broad Street
;;                   Columbus, Ohio 43213 USA
;;
;; Maintainer: Gary F. Ellison <Gary.F.Ellison@ATT.COM>
;; Created: Fri Jun 19
;; Version: 2.6
;; Keywords: languages, shell, korn, bourne, sh, ksh, bash, unix
;;
;; Delta On        : 2/23/95
;; Last Modified By: Gary Ellison
;; Last Modified On: Thu Feb 23 11:32:03 1995
;; Update Count    : 33
;; Status          : Highly Functional
;;

;;; Commentary:

;;
;; Description:
;;   sh, ksh, and bash script editing commands for emacs.
;; 
;; Installation:
;;   Put protogen-mode.el in some directory in your load-path.
;;   Refer to the installation section of protogen-mode's function definition.
;;
;; Usage:
;;   This major mode assists protogen specification writers with indentation
;;   control and control structure construct matching in much the same
;;   fashion as other programming language modes. Invoke describe-mode
;;   for more information.
;; 
;; Bugs:
;;   When the protogen-align-to-keyword is non-nil and the nester
;;   is a multi-command expression with a compound command
;;   the lines following the compound end will align incorrectly
;;   to the compound command instead of it's current indentation.
;;   The fix will probably require the detection of syntax elements
;;   in the nesting line.
;;   
;;   Function ending brace "}" must be on a separate line for indent-line
;;   to do the right thing.
;;
;;   Explicit function definition matching will proclaim in the minibuffer
;;   "No matching compound command" followed by "Matched ... "
;;
;;   indent-for-comment fails to recognize a comment starting in column 0,
;;   hence it moves the comment-start in comment-column.

;;; Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; HISTORY 
;; 31-July-96 		Bill Davies <daviesw@agcs.com>
;;    Original creation. Copied from ksh-mode.el and hacked to fit. 
;; 
;; Still to be done:
;;
;;  - remove unused scraps from ksh 
;;
;;  - add support for %{ .. %} include blocks 
;;
;;  - add support for optional terminating semicolon ';' 
;;     ** DONE but it is outdented .. may want to add option to 
;;            indent OR outdent 
;;
;;  - add support for array indentation [...] 
;;    cuddle the '[' and ']' right up to the value and insert a 
;;    space between the field name and the opening '[' 
;;
;;  - add support for a 'field column' where the field names 
;;    start    <type>  <field-name> 
;;
;;  - add support for a 'comment column' where end-of-line comments 
;;    start 
;;
;;  - add hot keys (and menu) support for operations like 
;;    next-message, prev-message, etc 
;;
;;  - adapt the completion support (bottom of file) to ProtoGen 
;;
;; ----- rest from KSH -- 
;; 8-Aug-95		Jack Repenning <jackr@sgi.com>
;;    Fix documentation of `protogen-align-to-keyword' to conform to the 23
;;    Feb default change.  Search for keywords obeying case, since the
;;    shell does.
;;
;; 23-Feb-1995		Gary Ellison	
;;    Merged Jonathan Stigelman <Stig@hackvan.com> into 2.5 souce.
;;
;; 23 Feb 1995          Jonathan Stigelman <Stig@hackvan.com>
;;    Reshuffled documentation to make the format more consistant with other
;;    elisp.  Added autoload and removed autoloading instructions from the
;;    protogen-mode docstring.  Changed default value for `protogen-align-to-keyword'
;;    to nil because it doesn't work properly.
;;
;; 2-Aug-1994		Gary Ellison	
;;    Last Modified: Mon Jun 13 16:52:55 1994 #29 (Gary Ellison)
;;    - Syntax table modifications to better support sexp navigation and
;;      parsing.
;;    - Fixed keyword regexps. Keywords were not being recoginized on the
;;      same line as " ' `.
;;
;; 13-Jun-1994		Gary Ellison	
;;    Last Modified: Wed Mar 30 14:12:26 1994 #28 (Gary Ellison)
;;    - Minor excursion problem fixed in protogen-indent-command.
;;
;; 30-Mar-1994		Gary Ellison	
;;    Last Modified: Fri Mar 25 15:42:29 1994 #25 (Gary Ellison)
;;    - Implement user customizable protogen-comment-regexp.
;;    - Make the keyword vs line indentation alignment customizable
;;      by calling protogen-align-to-keyword based on variable of same
;;      name. (If the code is obfuscated or convoluted I can attribute
;;      this to a severe head cold and not malice :)
;;
;; 25-Mar-1994		Gary Ellison	
;;    Last Modified: Fri Feb  4 13:06:30 1994 #23 (Gary Ellison)
;;    - Nest relative to the line indentation not the keywords
;;      column.
;;
;; 4-Feb-1994		Gary Ellison	
;;    Last Modified: Wed Nov 10 10:03:01 1993 #18 (Gary Ellison)
;;    - Add direct support for font-lock-mode. Thanks Espen Skoglund
;;      for the regular expressions.
;;
;; 10-Nov-1993		Gary Ellison	
;;    Last Modified: Tue Oct 12 15:23:06 1993 #17 (Gary Ellison)
;;    Fix message on protogen-match-and-tell to not get invalid format
;;    when a % appears in the string.
;;
;; 12-Oct-1993		Espen Skoglund <espensk@stud.cs.uit.no>.
;;    Last Modified: Tue Oct 12 15:03:01 1993 #16 (Gary Ellison)
;;    Apply Line continuation patch supplied by Espen Skoglund
;;
;; 1-Sep-1993		Gary Ellison	
;;    Last Modified: Tue Aug 17 17:18:18 1993 #14 (Gary Ellison)
;;    Get rid of this-line hack in protogen-get-nester-column.
;;
;; 17-Aug-1993		Gary Ellison	
;;    Last Modified: Mon Jun 21 14:00:43 1993 #13 (Gary Ellison)
;;    Code uses builtin current-indentation instead of lisp defun
;;    protogen-indentation-on-this-line (thanks to Tom Tromey).
;;    More and better doc strings.
;;
;; 5-Aug-1993		Tom Tromey <tromey@cns.caltech.edu>
;;    Last Modified: Thu Aug  5 11:09:12 1993 #12 (Tom Tromey)
;;    protogen-indent-region skips blank lines.  Uses let binding instead
;;    of setq.  No longer marks buffer modified if indentation
;;    doesn't change. 
;;
;; 21-Jun-1993		Gary Ellison	
;;    Last Modified: Mon Mar 29 15:05:34 1993 #11 (Gary Ellison)
;;    Use make-local-variables instead of make-variables-buffer-local
;;    protogen-indent now supports nil (keyword aligned) or number (offset)
;;    Support protogen-tab-always-indent feature
;;    Variables offsetting indentation renamed to better reflect their
;;    role.
;;    Integrate keyword completion feature supplied by
;;    Haavard Rue <hrue@imf.unit.no>.
;;
;; 29-Mar-1993		Gary Ellison	
;;    Last Modified: Tue Sep 29 16:14:02 1992 #10 (Gary Ellison)
;;    Integrate line continuation patch supplied by
;;    Haavard Rue <hrue@imf.unit.no>
;;    Name back to protogen-mode to avoid confusion with sh-mode
;;    by Thomas W. Strong, Jr. <strong+@cmu.edu>.
;;
;; 29-Sep-1992		Gary Ellison	
;;    Last Modified: Wed Sep  2 08:51:40 1992 #9 (Gary Ellison)
;;    Full support of protogen88 case items. 
;;    Align statements under "do" and "then" keywords one position 
;;    past the keyword.
;;
;; 2-Sep-1992		Gary Ellison	
;;    Last Modified: Tue Aug  4 14:34:35 1992 #8 (Gary Ellison)
;;    Use make-variable-buffer-local instead of make-local-variable
;;    Get rid of superflous protogen-default variables.
;;    Use end of word match \b for "then", "do", "else", "elif"
;;    Support process substitution lists and exclude protogen 88 case items
;;    Use default-tab-width for indentation defaults.
;;    Moved installation instructions to the mode level documentation 
;;    section.
;;    Fixed auto-mode-alist documentation.
;;
;; 24-Jul-1992		Gary Ellison	
;;    Last Modified: Fri Jul 24 09:45:11 1992 #7 (Gary Ellison)
;;    Modified protogen-indent-region to use marker versus fixed end point.
;;    comment-start-skip regexp no longer fooled by parameter substitution.
;;    Added constant protogen-mode-version.
;;
;; 21-Jul-1992		Gary Ellison	
;;    Last Modified: Tue Jul 21 15:53:57 1992 #6 (Gary Ellison)
;;    Indent with tabs instead of spaces.
;;    Can handle just about all styles.
;;    Anti-newline in REs.
;;    Word delim "\b" in REs
;;    More syntax entries.
;;    Variables with regexp suffix abbreviated to re
;;    Better } handling
;;    Implemented minimal indent-region-function
;;    Mode documentation corrected.
;;    Minor lisp source format changes.
;;    
;; 29-Jun-1992		Gary Ellison	
;;    Last Modified: Mon Jun 29 15:39:35 1992 #5 (Gary Ellison)
;;    Optimize line-to-string
;;    Implicit/Explicit functions aok
;;    More indentation variables
;;    Superfluous defun killed.
;;    renamed to sh-mode
;;    
;; 22-Jun-1992          Gary Ellison
;;    Last Modified: Mon Jun 22 15:01:14 1992 #4 (Gary Ellison)
;;    Cleanup pre att.emacs posting
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 17:19:14 1992 #3 (Gary Ellison)
;;    Minimal case indent handling
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 16:23:26 1992 #2 (Gary Ellison)
;;    Nesting handled except for case statement
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 10:03:07 1992 #1 (Gary Ellison)
;;    Conception of this mode.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst protogen-mode-version "1.1"
  "*Version numbers of this version of protogen-mode")

;;
;; Variables controlling indentation style
;;

(defvar protogen-indent 3
  ;; perhaps c-basic-offset would be okay to use as a default, but using
  ;; default-tab-width as the default is ridiculous --Stig
  "*Indentation of protogen statements with respect to containing block.
A value of nil indicates  TBD [FIX this] compound list keyword \(\"do\" and \"then\"\) alignment.")

;; FIX - not used
 (defvar protogen-case-item-offset protogen-indent
  "*Additional indentation for case items within a case statement.")

;; FIX - not used
 (defvar protogen-case-indent nil
  "*Additional indentation for statements under case items.")

;; FIX - not used
(defvar protogen-group-offset (- protogen-indent)
  "*Additional indentation for keywords \"do\" and \"then\".")

;; FIX - not used
(defvar protogen-brace-offset 0
  "*Additional indentation of \"{\" under functions or brace groupings.")

;; FIX - not used
(defvar protogen-multiline-offset 1
  "*Additional indentation of line that is preceded of a line ending with a
\\ to make it continue on next line.")

;; FIX - not used
(defvar protogen-match-and-tell t
  "*If non-nil echo in the minibuffer the matching compound command
for the \"done\", \"}\", \"fi\", or \"esac\". ")

(defvar protogen-tab-always-indent t
  "*Controls the operation of the TAB key. If t (the default), always
reindent the current line.  If nil, indent the current line only if
point is at the left margin or in the line's indentation; otherwise
insert a tab.")

(defvar protogen-align-to-keyword nil
  ;; #### - this is broken, so it should be disabled by default --Stig
  "*Controls whether nested constructs align from the keyword or
the current indentation. If non-nil, indentation will be relative to
the column the keyword starts. If nil, indentation will be relative to
the current indentation of the line the keyword is on.
The default value is nil.
The non-nil case doesn't work very well.")

(defvar protogen-comment-regexp "//"
  "*Regular expression used to recognize comments. Customize to support
protogen-like languages.")

(defun protogen-current-indentation ()
  nil
  )
;;
(fset 'protogen-current-indentation 'current-column)

;;
;; Variables controlling completion
(defvar protogen-completion-list '())
(make-variable-buffer-local 'protogen-completion-list)
(set-default 'protogen-completion-list  '())

;;
;; -type-  : type number, 0:misc, 1:variable, 2:function
;; -regexp-: regexp used to parse the script
;; -match- : used by match-beginning/end to pickup target
;;
(defvar protogen-completion-type-misc 0)
(defvar protogen-completion-regexp-var "\\([A-Za-z_0-9]+\\)=")
(defvar protogen-completion-type-var 1)
(defvar protogen-completion-match-var 1) 
(defvar protogen-completion-regexp-var2 "\\$\\({\\|{#\\)?\\([A-Za-z_0-9]+\\)[#%:}]?")
(defvar protogen-completion-match-var2 2)
(defvar protogen-completion-regexp-function
  "\\(function\\)?[ \t]*\\([A-Za-z_0-9]+\\)[ \t]*([ \t]*)")
(defvar protogen-completion-type-function 2)
(defvar protogen-completion-match-function 2)

;;
;; Variable controlling fontification
;;

;; FIX: add %{ .. %} block highlighting 

(defvar protogen-keywords 
  '( "%apikey"
     "%pragma" 
     "%agent"
     "%const"
     "%type"
     "%enum"
     "%alias"
     "%message"
     "%/"
     "is"
     "primitive"
     "extendable"
     "extends"
     "BYTE" "WORD" "DWORD" "STRING" 
     "PREFIX" "TAG" "NOTAG" ))

(defconst protogen-font-lock-keywords
      (list

       ;; Fontify keywords
       (cons (concat
	      "\\(\\<"
	      (mapconcat 'identity protogen-keywords "\\>\\|\\<")
	      "\\>\\)")
	     1)

       ;; Fontify symbol names
       '("\\<\\%/[ \t]+\\(.*\\)"  1 font-lock-doc-string-face)
       '("\\<\\%apikey[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"  1 font-lock-doc-string-face)
       '("\\<\\%pragma[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"  1 font-lock-doc-string-face)
       '("\\<\\%const[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"   1 font-lock-type-face)
       '("\\<\\%alias[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"   1 font-lock-type-face)
       '("\\<\\%enum[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"    1 font-lock-type-face)
       '("\\<\\%type[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"    1 font-lock-type-face)
       '("\\<\\%message[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)" 1 font-lock-function-name-face)

       ;; Fontify transfer specs
       '("^[ \t]*\\(\\@\\)"                                     1 font-lock-doc-string-face t)
       '("^[ \t]*\\@[ \t]+[A-Za-z][A-Za-z0-9_]+[ \t]+\\(<-\\)"  1 font-lock-doc-string-face t)
       '("^[ \t]*\\@[ \t]+[A-Za-z][A-Za-z0-9_]+[ \t]+\\(<->\\)" 1 font-lock-doc-string-face t)
       '("^[ \t]*\\@[ \t]+[A-Za-z][A-Za-z0-9_]+[ \t]+\\(->\\)"  1 font-lock-doc-string-face t)

       ;; Fontify include blocks 
       '("^[ \t]*\\(\\%{\\)"  1 font-lock-keyword-face t)
       '("^[ \t]*\\(\\%}\\)"  1 font-lock-keyword-face t)
       ;; FIX: '("^[ \t]*\\%{[ \t]*\n.*\\%}\\)"  1 font-lock-doc-string-face t)

       ))

(put 'protogen-mode 'font-lock-keywords 'protogen-font-lock-keywords)

;; XEmacs change -- This can incorrectly set some Perl scripts to
;; protogen-mode.  It also won't work for some other shells which protogen-mode
;; nominally works with.
;(defun protogen-check-hook ()
;    (save-excursion
;     (save-restriction
;       (widen)
;       (goto-char (point-min))
;       (cond ((looking-at "#![ \t]*/.*/k?sh[ \t]*")
;	      (protogen-mode))))))
;
;(add-hook 'find-file-hooks 'protogen-check-hook)

;;
;; Context/indentation regular expressions
;; 
;; indenting expressions
;;

(defconst protogen-apikey-re     "^[ \t]*%apikey[ \t]+"
  "*Regexp used to locate %apikey statement" )

(defconst protogen-pragma-re     "^[ \t]*%pragma[ \t]+"
  "*Regexp used to locate %pragma statement" )

(defconst protogen-agent-re     "^[ \t]*%agent[ \t]+"
  "*Regexp used to locate %agent statement" )

(defconst protogen-const-re     "^[ \t]*%const[ \t]+"
  "*Regexp used to locate %const statement" )

(defconst protogen-alias-re     "^[ \t]*%alias[ \t]+"
  "*Regexp used to locate %alias statement" )

(defconst protogen-enum-re     "^[ \t]*%enum[ \t]+"
  "*Regexp used to locate %enum statement" )

(defconst protogen-type-re     "^[ \t]*%type[ \t]+"
  "*Regexp used to locate %type statement" )

(defconst protogen-message-re     "^[ \t]*%message[ \t]+"
  "*Regexp used to locate message definition" )

(defconst protogen-passthru-re     "^[ \t]*%//[ \t]+"
  "*Regexp used to locate passthru comment definition" )

(defconst protogen-terminator-re  "^[ \t]*;[ \t]*"
  "*Regexp used to locate compound statement terminator" )

(defconst protogen-blank-line-re      "^\\s *$"
  "*Regexp used to locate line containinf a single symbol" )

(defconst protogen-keywords-re
  "^[^/\n]*\\b\\(%message\\|%type\\%enum\\)\\b"
  "*Regexp used to detect compound command keywords: %message, %type, %enum")

(defconst protogen-statement-keywords-re
  "^[^/\n]*\\b\\(%//\\|%apikey\\|%pragma\\|%agent\\|%const\\|%type\\|%alias\\|%enum\\|%message\\)\\b"
  "*Regexp used to detect ProtoGen statement keywords: %apikey, %pragma, %agent, %const,
%type, %alias, %enum, %message")





;; -- from KSH --

(defconst protogen-then-do-re     "^[^#\n]*\\s\"*\\b\\(then\\|do\\)\\b"
  "*Regexp used to locate grouping keywords: \"then\" and \"do\"" )

(defconst protogen-do-re          "^\\s *\\bdo\\(\\b\\|$\\)"
  "*Regexp used to match keyword: do")

(defconst protogen-then-re        "^\\s *\\bthen\\(\\b\\|$\\)"
  "*Regexp used to match keyword: then")

;;
;; Structure starting/indenting keywords
;;
(defconst protogen-else-re           "^\\s *\\belse\\(\\b\\|$\\)"
  "*Regexp used to match keyword: else")

(defconst protogen-elif-re           "^\\s *\\belif\\(\\b\\|$\\)"
  "*Regexp used to match keyword: elif")

(defconst protogen-brace-re           "^\\S>*{[ \t\n]"
  "*Regexp used to match syntactic entity: { ")

(defconst protogen-case-item-end-re           "^\\S>*;;[ \t\n]"
  "*Regexp used to match case item end syntactic entity: ;;")

(defconst protogen-if-re         "^[^#\n]*\\s\"*\\b\\(if\\)\\b"
  "*Regexp used to match keyword: if")

(defconst protogen-iteration-keywords-re 
  "^[^#\n]*\\s\"*\\b\\(while\\|for\\|until\\|select\\)\\b"
  "*Match one of the keywords: while, until, for, select")

(defconst protogen-case-re           "^[^#\n]*\\s\"*\\b\\(case\\)\\b"
  "*Regexp used to match keyword: case")

(defconst protogen-explicit-func-re
  "^\\s *\\(function\\s [a-zA-z_][a-zA-Z0-1_]*\\)\\b"
  "*Match an explicit function definition: function name")

(defconst protogen-implicit-func-re
  "^\\s *\\([a-zA-z_][a-zA-Z0-1_]*\\)\\s *()\\s *"
  "*Match an implicit function definition: name ()")

(defconst protogen-func-brace-re "^\\s *\\(.*{\\)[ \t\n]+"
  "*Match a implicit function definition brace: name { ")

;;
;; indenting 
(defconst protogen-case-item-re           "^[^#\n]*\\s\"*\\()\\)"
  "*Regexp used to match case-items including protogen88")

(defconst protogen-paren-re           "^[^#\n]*\\s\"*)[ \t\n]+"
  "*Regexp used to match compound list & case items")

;;
;; structure ending keyword regular expressions
(defconst protogen-fi-re            "^\\s *fi\\b"
  "*Regexp used to match keyword: fi")

(defconst protogen-esac-re          "^\\s *esac\\b"
  "*Regexp used to match keyword: esac")

(defconst protogen-done-re          "^\\s *done\\b"
  "*Regexp used to match keyword: done")

(defconst protogen-brace-end-re  "^\\s *}\\s *"
  "*Regexp used to match function brace-groups")

(defconst protogen-multiline-re "^.*\\\\$"
  "*Regexp used to match a line with a statement using more lines.")

;;
;;
;; Create mode specific tables
(defvar protogen-mode-syntax-table nil
  "Syntax table used while in protogen mode.")
(if protogen-mode-syntax-table
    ()
  (setq protogen-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\' "\"" protogen-mode-syntax-table)
  (modify-syntax-entry ?`  "\"" protogen-mode-syntax-table)
  (modify-syntax-entry ?\n ">" protogen-mode-syntax-table)
  (modify-syntax-entry ?\f ">" protogen-mode-syntax-table)
  (modify-syntax-entry ?#  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?_  "w" protogen-mode-syntax-table)
  (modify-syntax-entry ?<  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?>  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?&  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?|  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?$  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?%  "w" protogen-mode-syntax-table)
  (modify-syntax-entry ?=  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?/  ". 12" protogen-mode-syntax-table)
  (modify-syntax-entry ?+  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?*  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?-  "." protogen-mode-syntax-table)
  (modify-syntax-entry ?\; "." protogen-mode-syntax-table)
  )

(defvar protogen-mode-abbrev-table nil
  "Abbrev table used while in protogen mode.")
(define-abbrev-table 'protogen-mode-abbrev-table ())

(defvar protogen-mode-map nil 
  "Keymap used in protogen mode")

(if protogen-mode-map
    ()
  (setq protogen-mode-map (make-sparse-keymap))
  (define-key protogen-mode-map "\t"      'protogen-indent-command)
  (define-key protogen-mode-map "\n"      'reindent-then-newline-and-indent)
  (define-key protogen-mode-map '[return] 'reindent-then-newline-and-indent)
;;  (define-key protogen-mode-map "\t"    'protogen-indent-line)
;;  (define-key protogen-mode-map "\177"  'backward-delete-char-untabify)
  (define-key protogen-mode-map "\C-j"    'reindent-then-newline-and-indent)
  (define-key protogen-mode-map "\e\t"    'protogen-complete-symbol)
  (define-key protogen-mode-map "\C-c\t"  'protogen-completion-init-and-pickup)
  )


;;;###autoload
(defun protogen-mode ()
  "protogen-mode 2.6 - Major mode for editing (Bourne, Korn or Bourne again)
shell scripts.
Special key bindings and commands:
\\{protogen-mode-map}
Variables controlling indentation style:
protogen-indent
    Indentation of protogen statements with respect to containing block.
    Default value is 2.
protogen-case-indent
    Additional indentation for statements under case items.
    Default value is nil which will align the statements one position 
    past the \")\" of the pattern.
protogen-case-item-offset
    Additional indentation for case items within a case statement.
    Default value is 2.
protogen-group-offset
    Additional indentation for keywords \"do\" and \"then\".
    Default value is -2.
protogen-brace-offset
    Additional indentation of \"{\" under functions or brace groupings.
    Default value is 0.
protogen-multiline-offset
   Additional indentation of line that is preceded of a line ending with a
   \\ to make it continue on next line.
protogen-tab-always-indent
    Controls the operation of the TAB key. If t (the default), always
    reindent the current line.  If nil, indent the current line only if
    point is at the left margin or in the line's indentation; otherwise
    insert a tab.
protogen-match-and-tell
    If non-nil echo in the minibuffer the matching compound command
    for the \"done\", \"}\", \"fi\", or \"esac\". Default value is t.

protogen-align-to-keyword
    Controls whether nested constructs align from the keyword or
    the current indentation. If non-nil, indentation will be relative to
    the column the keyword starts. If nil, indentation will be relative to
    the current indentation of the line the keyword is on.
    The default value is non-nil.

protogen-comment-regexp
  Regular expression used to recognize comments. Customize to support
  protogen-like languages. Default value is \"\^\\\\s *#\".

Style Guide.
 By setting
    (setq protogen-indent default-tab-width)
    (setq protogen-group-offset 0)

    The following style is obtained:

    if [ -z $foo ]
	    then
		    bar    # <-- protogen-group-offset is additive to protogen-indent
		    foo
    fi

 By setting
    (setq protogen-indent default-tab-width)
    (setq protogen-group-offset (- 0 protogen-indent))

    The following style is obtained:

    if [ -z $foo ]
    then
	    bar
	    foo
    fi

 By setting
    (setq protogen-case-item-offset 1)
    (setq protogen-case-indent nil)

    The following style is obtained:

    case x in *
     foo) bar           # <-- protogen-case-item-offset
          baz;;         # <-- protogen-case-indent aligns with \")\"
     foobar) foo
             bar;;
    esac

 By setting
    (setq protogen-case-item-offset 1)
    (setq protogen-case-indent 6)

    The following style is obtained:

    case x in *
     foo) bar           # <-- protogen-case-item-offset
           baz;;        # <-- protogen-case-indent
     foobar) foo
           bar;;
    esac
    

Installation:
  Put protogen-mode.el in some directory in your load-path.
  Put the following forms in your .emacs file.

 (setq auto-mode-alist
      (append auto-mode-alist
              (list
               '(\"\\\\.pro$\" . protogen-mode))))

 (setq protogen-mode-hook
      (function (lambda ()
         (font-lock-mode 1)             ;; font-lock the buffer
         (setq protogen-indent 8)
	 (setq protogen-group-offset -8))
	 (setq protogen-brace-offset -8)   
         (setq protogen-tab-always-indent t)
         (setq protogen-match-and-tell t)
         (setq protogen-align-to-keyword t)	;; Turn on keyword alignment
	 )))"
  (interactive)
  (kill-all-local-variables)

  (use-local-map protogen-mode-map)

  (setq major-mode 'protogen-mode)
  (setq mode-name "Protogen")

  (setq local-abbrev-table protogen-mode-abbrev-table)
  (set-syntax-table protogen-mode-syntax-table)

  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'protogen-indent-line)

  (make-local-variable 'indent-region-function)
  (setq indent-region-function 'protogen-indent-region)

  (make-local-variable 'comment-start)
  (setq comment-start "// ")

  (make-local-variable 'comment-end)
  (setq comment-end "")

  (make-local-variable 'comment-column)
  (setq comment-column 32)

  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "// *")
  ;;
  ;; config font-lock mode
  (make-local-variable 'font-lock-keywords) 
  (setq font-lock-keywords protogen-font-lock-keywords)
  ;;
  ;; Let the user customize
  (run-hooks 'protogen-mode-hook)
  (if (not protogen-align-to-keyword)
      (protogen-align-to-keyword -1)
    )
  ) ;; defun

;;
;; Support functions

(defun protogen-align-to-keyword (&optional arg)
  "Toggle value of protogen-align-to-keyword and rebind the protogen-current-indentation
function. With arg, force alignment to keyword if and only if arg is positive."
  (interactive)
  (if (null arg)			;just toggle
      (cond ((not protogen-align-to-keyword)
	     (setq protogen-align-to-keyword t)
	     (fset 'protogen-current-indentation 'current-column))
	    (t
	     (setq protogen-align-to-keyword nil)
	     (fset 'protogen-current-indentation 'current-indentation))
	    )
    (cond ((natnump arg)
	   (setq protogen-align-to-keyword t)
	   (fset 'protogen-current-indentation 'current-column))
	  (t
	   (setq protogen-align-to-keyword nil)
	   (fset 'protogen-current-indentation 'current-indentation))
	  ))
  )

(defun protogen-current-line ()
  "Return the vertical position of point in the buffer.
Top line is 1."
  (+ (count-lines (point-min) (point))
     (if (= (current-column) 0) 1 0))
  )


(defun protogen-line-to-string ()
  "From point, construct a string from all characters on
current line"
  (skip-chars-forward " \t") ;; skip tabs as well as spaces
  (buffer-substring (point)
                    (progn
                      (end-of-line 1)
                      (point))))


;;
;; Search backwards in the file to find the scope. 
;; Collects indentation level of first non-blank (non-continued) line 
;; 

(defun protogen-get-nest-level ()
  "Return a 2 element list (nest-level nest-line) describing where the
current line should nest."
  (let ((case-fold-search)
    	(level))
    (save-excursion
      (forward-line -1)
      (while (and (not (bobp))
		  (null level))

        ;; if not looking at blank line AND 
        ;; the current line is not part of a continued
        ;; multiline 
        ;;  then level := current-indent.current-line  
        ;; 
	;; (if (and (not (looking-at "^\\s *$"))
	(if (and (not (looking-at protogen-blank-line-re))
 		 (not (save-excursion
 			(forward-line -1) ;; go back 1 line 
 			(beginning-of-line)
			(looking-at protogen-multiline-re)))
		 (not (looking-at protogen-comment-regexp)))
	    (setq level (cons (current-indentation)
			      (protogen-current-line)))
	  (forward-line -1)
	  );; if
	);; while
      (if (null level)
	  (cons (current-indentation) (protogen-current-line))
	level)
      )
    )
  )

(defun protogen-looking-at-compound-list ()
  "Return true if current line contains compound list initiating keyword"
  (or 
   (looking-at protogen-do-re)
   (looking-at protogen-then-re)
   ) ;; or
  ) ;; defun

(defun protogen-looking-at-case-item ()
  "Return true if current line is a case-item .vs. paren compound list"
  (save-excursion
    (beginning-of-line)
    ;;
    ;; Handle paren indentation constructs for this line
    (cond ((looking-at protogen-paren-re)
	   (goto-line (cdr (protogen-get-nest-level)))
	   ;;
	   ;; The question is whether this is really a case item or just
	   ;; parenthesized compound list.
	   (cond ((or (looking-at protogen-case-re)
		      (looking-at protogen-case-item-end-re)))
		 ;;
		 ;; turns out to be a parenthesized compound list
		 ;; so propigate the nil for cond
		 )
	   ))
    )
  ) ;; defun

(defun protogen-get-case-indent ()
  "Return the column of the closest open case statement"
  (save-excursion
    (let (
	  (nest-list (protogen-get-compound-level protogen-case-re protogen-esac-re (point)))
	  )
      (if (null nest-list)
	  (progn 
	    (if protogen-match-and-tell
		(message "No matching case for ;;"))
	    0)
	(car nest-list)))
    )
  )

;;
;; Functions which make this mode what it is
;;

(defun protogen-get-nester-column (nest-line)
  "Return the column to indent to with respect to nest-line taking 
into consideration keywords and other nesting constructs."
  (save-excursion 
    (let ((fence-post)
	  (nester-column)
	  (case-fold-search)
	  (start-line (protogen-current-line)))
      ;;
      ;; Handle case item indentation constructs for this line
      (cond ((protogen-looking-at-case-item)
	     (save-excursion
	       (goto-line nest-line)
	       (let ((fence-post (save-excursion (end-of-line) (point))))
		 ;;
		 ;; Now know there is a case-item so detect whether
		 ;; it is first under case, just another case-item, or
		 ;; a case-item and case-item-end all rolled together.
		 ;;
		 (cond ((re-search-forward protogen-case-re fence-post t)
			(goto-char (match-beginning 1))
			(+ (protogen-current-indentation) protogen-case-item-offset))

		       ((protogen-looking-at-case-item)
			(current-indentation))

		       ((looking-at protogen-case-item-end-re)
			(end-of-line)
			(+ (protogen-get-case-indent) protogen-case-item-offset))
		       )
		 )))
	    (t;; Not a case-item.  What to do relative to the nest-line?
	     (save-excursion
	       (goto-line nest-line)
	       (setq fence-post (save-excursion (end-of-line) (point)))
	       (setq nester-column
		     (save-excursion
		       (cond
			;;
			;; Check if we are in a continued statement
			((and (looking-at protogen-multiline-re)
			      (save-excursion
				(goto-line (1- start-line))
				(looking-at protogen-multiline-re)))
			 (+ (current-indentation) protogen-multiline-offset))

			;; In order to locate the column of the keyword,
			;; which might be embedded within a case-item,
			;; it is necessary to use re-search-forward.
			;; Search by literal case, since shell is
			;; case-sensitive.
			((re-search-forward protogen-keywords-re fence-post t)
			 (goto-char (match-beginning 1))
			 (if (looking-at protogen-case-re)
			     (+ (protogen-current-indentation) protogen-case-item-offset)
			   (+ (protogen-current-indentation)
			      (if (null protogen-indent)
				  2 protogen-indent)
			      )))

			;;----------------------------------------------------
                        ;; from KSH 
			;;----------------------------------------------------
                        ;;((re-search-forward protogen-then-do-re fence-post t)
			;; (if (null protogen-indent)
			;;     (progn 
			;;       (goto-char (match-end 1))
			;;       (+ (protogen-current-indentation) 1))
			;;   (progn
			;;     (goto-char (match-beginning 1))
			;;     (+ (protogen-current-indentation) protogen-indent))
			;;   ))
			;;----------------------------------------------------

			((looking-at protogen-brace-re)
			 (+ (current-indentation)
			    (if (null protogen-indent)
				2 protogen-indent)
			    ))

			;;
			;; Forces functions to first column
			((or (looking-at protogen-implicit-func-re)
			     (looking-at protogen-explicit-func-re))
			 (if (looking-at protogen-func-brace-re)
			     (if (null protogen-indent)
				 2 protogen-indent)
			   protogen-brace-offset))

			;;
			;; Need to first detect the end of a case-item
			((looking-at protogen-case-item-end-re)
			 (end-of-line)
			 (+ (protogen-get-case-indent) protogen-case-item-offset))
			;;
			;; Now detect first statement under a case item
			((protogen-looking-at-case-item)
			 (if (null protogen-case-indent)
			     (progn
			       (re-search-forward protogen-case-item-re fence-post t)
			       (goto-char (match-end 1))
			       (+ (current-column) 1))
			   (+ (current-indentation) protogen-case-indent)))

			;;----------------------------------------------------
                        ;; Indent for %message 
			;;----------------------------------------------------
			((looking-at protogen-message-re)
			 (+ (current-indentation)
			    (if (null protogen-indent)
				2 protogen-indent)
			    ))
			;;----------------------------------------------------
                        ;; Indent for %type 
			;;----------------------------------------------------
			((looking-at protogen-type-re)
			 (+ (current-indentation)
			    (if (null protogen-indent)
				2 protogen-indent)
			    ))
			;;----------------------------------------------------
                        ;; Indent for %enum
			;;----------------------------------------------------
			((looking-at protogen-enum-re)
			 (+ (current-indentation)
			    (if (null protogen-indent)
				2 protogen-indent)
			    ))
			;;
			;; This is hosed when using current-column
			;; and there is a multi-command expression as the
			;; nester.
			(t (current-indentation)))
		       )
		     ));; excursion over
	     ;;
	     ;; Handle additional indentation constructs for this line
	     (cond ((protogen-looking-at-compound-list)
		    (+ nester-column protogen-group-offset))
		   ((looking-at protogen-brace-re)
		    (+ nester-column protogen-brace-offset))
		   (t nester-column))
	     );; Not a case-item
	    )
      );;let
    );; excursion
  );; defun

(defun protogen-indent-command ()
  "Indent current line relative to containing block and allow for
protogen-tab-always-indent customization"
  (interactive)
  (let (case-fold-search)
    (cond ((save-excursion
	     (skip-chars-backward " \t")
	     (bolp))
	   (protogen-indent-line))
	  (protogen-tab-always-indent
	   (save-excursion
	     (protogen-indent-line)))
	  (t (insert-tab))
	  ))
  )


(defun protogen-indent-line ()
  "Indent current line as far as it should go according
to the syntax/context"
  (interactive)
  (let (case-fold-search)
    (save-excursion
      (beginning-of-line)
      (if (bobp)
	  nil
	;;
	;; Align this line to current nesting level
	(let*
	    (
	     (level-list (protogen-get-nest-level)) ; Where to nest against
	     ;;           (last-line-level (car level-list))
	     (this-line-level (current-indentation))
	     (nester-column (protogen-get-nester-column (cdr level-list)))
	     (struct-match (protogen-match-structure-and-reindent))
	     )
	  (if struct-match
	      (setq nester-column struct-match))
	  (if (eq nester-column this-line-level)
	      nil
	    (beginning-of-line)
	    (let ((beg (point)))
	      (back-to-indentation)
	      (delete-region beg (point)))
	    (indent-to nester-column))
	  );; let*
	);; if
      );; excursion
    ;;
    ;; Position point on this line
    (let*
	(
	 (this-line-level (current-indentation))
	 (this-bol (save-excursion
		     (beginning-of-line)
		     (point)))
	 (this-point (- (point) this-bol))
	 )
      (cond ((> this-line-level this-point);; point in initial white space
	     (back-to-indentation))
	    (t nil)
	    );; cond
      );; let*
    );; let
  );; defun


(defun protogen-match-indent-level (begin-re end-re)
  "Match the compound command and indent. Return nil on no match,
indentation to use for this line otherwise."
  (interactive)
  (let* ((case-fold-search)
	 (nest-list 
	  (save-excursion
	    (protogen-get-compound-level begin-re end-re (point))
	    ))
	 ) ;; bindings
    (if (null nest-list)
	(progn
	  (if protogen-match-and-tell
	      (message "No matching compound command"))
	  nil) ;; Propagate a miss.
      (let* (
	     (nest-level (car nest-list))
	     (match-line (cdr nest-list))
	     ) ;; bindings
	(if protogen-match-and-tell
	    (save-excursion
	      (goto-line match-line)
	      (message "Matched ... %s" (protogen-line-to-string))
	      ) ;; excursion
	  ) ;; if protogen-match-and-tell
	nest-level ;;Propagate a hit.
	) ;; let*
      ) ;; if
    ) ;; let*
  ) ;; defun protogen-match-indent-level

(defun protogen-match-structure-and-reindent ()
  "If the current line matches one of the indenting keywords
or one of the control structure ending keywords then reindent. Also
if protogen-match-and-tell is non-nil the matching structure will echo in
the minibuffer"
  (interactive)
  (let (case-fold-search)
    (save-excursion
      (beginning-of-line)
      (cond ((looking-at protogen-else-re)
 	      (protogen-match-indent-level protogen-if-re protogen-fi-re))
	     ((looking-at protogen-elif-re)
	      (protogen-match-indent-level protogen-if-re protogen-fi-re))
	     ((looking-at protogen-fi-re)
	      (protogen-match-indent-level protogen-if-re protogen-fi-re))
	     ((looking-at protogen-done-re)
	      (protogen-match-indent-level protogen-iteration-keywords-re protogen-done-re))
	     ((looking-at protogen-esac-re)
	      (protogen-match-indent-level protogen-case-re protogen-esac-re))


	     ((looking-at protogen-apikey-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-apikey-re))

	     ((looking-at protogen-pragma-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-pragma-re))

	     ((looking-at protogen-agent-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-agent-re))

	     ((looking-at protogen-const-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-const-re))

	     ((looking-at protogen-alias-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-alias-re))

	     ((looking-at protogen-enum-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-enum-re))

	     ((looking-at protogen-type-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-type-re))

	     ((looking-at protogen-message-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-message-re))

	     ((looking-at protogen-terminator-re)
	      (protogen-match-indent-level protogen-statement-keywords-re protogen-terminator-re))

	    ;;
	    ((looking-at protogen-brace-end-re)
	     (cond
	      ((protogen-match-indent-level protogen-implicit-func-re protogen-brace-end-re))
	      ((protogen-match-indent-level protogen-explicit-func-re protogen-brace-end-re))
	      ((protogen-match-indent-level protogen-func-brace-re protogen-brace-end-re))
	      (t nil)))
	    (t nil)
	    );; cond
      )
    ))

(defun protogen-get-compound-level 
  (begin-re end-re anchor-point &optional balance-list)
  "Determine how much to indent this structure. Return a list (level line) 
of the matching compound command or nil if no match found."
  (let* 
      (;; Locate the next compound begin keyword bounded by point-min
       (match-point (if (re-search-backward begin-re (point-min) t)
			(match-beginning 1) 0))
       (nest-column (if (zerop match-point)
			1 
		      (progn
			(goto-char match-point)
			(protogen-current-indentation))))
       (nest-list (cons 0 0))    ;; sentinel cons since cdr is >= 1
       )
    (if (zerop match-point)
	nil ;; graceful exit from recursion
      (progn
	(if (nlistp balance-list)
	    (setq balance-list (list)))
	;; Now search forward from matching start keyword for end keyword
	(while (and (consp nest-list) (zerop (cdr nest-list))
		    (re-search-forward end-re anchor-point t))
	  (if (not (memq (point) balance-list))
	      (progn
		(setq balance-list (cons (point) balance-list))
		(goto-char match-point)  ;; beginning of compound cmd
		(setq nest-list
		      (protogen-get-compound-level begin-re end-re
					     anchor-point balance-list))
		)))

	(cond ((consp nest-list)
	       (if (zerop (cdr nest-list))
		 (progn
		   (goto-char match-point)
		   (cons nest-column (protogen-current-line)))
		 nest-list))
	      (t nil)
	      )
	)
      )
    )
  )


(defun protogen-indent-region (start end)
  "From start to end, indent each line."
  ;; The algorithm is just moving through the region line by line with
  ;; the match noise turned off.  Only modifies nonempty lines.
  (save-excursion
    (let (protogen-match-and-tell
	  (endmark (copy-marker end)))
      
      (goto-char start)
      (beginning-of-line)
      (setq start (point))
      (while (> (marker-position endmark) start)
	(if (not (and (bolp) (eolp)))
	    (protogen-indent-line))
	(forward-line 1)
	(setq start (point)))

      (set-marker endmark nil)
      )
    )
  )

;;
;; Completion code supplied by Haavard Rue <hrue@imf.unit.no>.
;;
;;
;; add a completion with a given type to the list
;;
(defun protogen-addto-alist (completion type)
  (setq protogen-completion-list
	(append protogen-completion-list
		(list (cons completion type)))))
;;
;; init the list and pickup all 
;;
(defun protogen-completion-init-and-pickup ()
  (interactive)
  (let (case-fold-search)
    (protogen-completion-list-init)
    (protogen-pickup-all)))

;;
;; init the list
;;
(defun protogen-completion-list-init ()
  (interactive)
  (setq protogen-completion-list
	(list
	 (cons "if"       protogen-completion-type-misc)
	 (cons "while"    protogen-completion-type-misc)
	 (cons "until"    protogen-completion-type-misc)
	 (cons "select"   protogen-completion-type-misc)
	 (cons "for"      protogen-completion-type-misc)
	 (cons "continue" protogen-completion-type-misc)
	 (cons "function" protogen-completion-type-misc)
	 (cons "fi"       protogen-completion-type-misc)
	 (cons "case"     protogen-completion-type-misc)
	 (cons "esac"     protogen-completion-type-misc)
	 (cons "break"    protogen-completion-type-misc)
	 (cons "exit"     protogen-completion-type-misc)
	 (cons "done"     protogen-completion-type-misc)
	 (cons "do"       protogen-completion-type-misc))))

(defun protogen-eol-point ()
  (save-excursion
    (end-of-line)
    (point)))

(defun protogen-bol-point ()
 (save-excursion
    (beginning-of-line)
    (point)))

(defun protogen-pickup-all ()
  "Pickup all completions in buffer."
  (interactive)
  (protogen-pickup-completion-driver (point-min) (point-max) t))

(defun protogen-pickup-this-line ()
  "Pickup all completions in current line."
  (interactive)
  (protogen-pickup-completion-driver (protogen-bol-point) (protogen-eol-point) nil))

(defun protogen-pickup-completion-driver (pmin pmax message)
  "Driver routine for protogen-pickup-completion."
  (if message
      (message "pickup completion..."))
  (let* (
	 (i1
	  (protogen-pickup-completion  protogen-completion-regexp-var
				 protogen-completion-type-var
				 protogen-completion-match-var
				 pmin pmax))
	 (i2
	  (protogen-pickup-completion  protogen-completion-regexp-var2
				 protogen-completion-type-var
				 protogen-completion-match-var2
				 pmin pmax))
	 (i3
	  (protogen-pickup-completion  protogen-completion-regexp-function
				 protogen-completion-type-function
				 protogen-completion-match-function
				 pmin pmax)))
    (if message
	(message "pickup %d variables and %d functions." (+ i1 i2) i3))))

(defun protogen-pickup-completion (regexp type match pmin pmax)
  "Pickup completion in region and addit to the list, if not already
there." 
  (let ((i 0) kw obj)
    (save-excursion
      (goto-char pmin)
      (while (and
	      (re-search-forward regexp pmax t)
	      (match-beginning match)
	      (setq kw  (buffer-substring
			 (match-beginning match)
			 (match-end match))))
	(progn
	  (setq obj (assoc kw protogen-completion-list))
	  (if (or (equal nil obj)
		  (and (not (equal nil obj))
		       (not (= type (cdr obj)))))
	      (progn
		(setq i (1+ i))
		(protogen-addto-alist kw type))))))
    i))

(defun protogen-complete-symbol ()
  "Perform completion."
  (interactive)
  (let* ((case-fold-search)
	 (end (point))
         (beg (unwind-protect
                  (save-excursion
                    (backward-sexp 1)
                    (while (= (char-syntax (following-char)) ?\')
                      (forward-char 1))
                    (point))))
         (pattern (buffer-substring beg end))
	 (predicate 
	  ;;
	  ;; ` or $( mark a function
	  ;;
	  (save-excursion
	    (goto-char beg)
	    (if (or
		 (save-excursion
		   (backward-char 1)
		   (looking-at "`"))
		 (save-excursion
		   (backward-char 2)
		   (looking-at "\\$(")))
		(function (lambda (sym)
			    (equal (cdr sym) protogen-completion-type-function)))
	      ;;
	      ;; a $, ${ or ${# mark a variable
	      ;;
	      (if (or
		   (save-excursion
		     (backward-char 1)
		     (looking-at "\\$"))
		   (save-excursion
		     (backward-char 2)
		     (looking-at "\\${"))
		   (save-excursion
		     (backward-char 3)
		     (looking-at "\\${#")))
		  (function (lambda (sym)
			      (equal (cdr sym)
				     protogen-completion-type-var)))
		;;
		;; don't know. use 'em all
		;;
		(function (lambda (sym) t))))))
	 ;;
	 (completion (try-completion pattern protogen-completion-list predicate)))
    ;;
    (cond ((eq completion t))
	  ;;
	  ;; oops, what is this ?
	  ;;
          ((null completion)
           (message "Can't find completion for \"%s\"" pattern))
	  ;;
	  ;; insert
	  ;;
          ((not (string= pattern completion))
           (delete-region beg end)
           (insert completion))
	  ;;
	  ;; write possible completion in the minibuffer,
	  ;; use this instead of a seperate buffer (usual)
	  ;;
          (t
           (let ((list (all-completions pattern protogen-completion-list predicate))
		 (string ""))
	     (while list
	       (progn
		 (setq string (concat string (format "%s " (car list))))
		 (setq list (cdr list))))
	     (message string))))))

(provide 'protogen-mode)
;;; protogen-mode.el ends here
