;; scl-mode.el
;;
;; Simulation command language mode 
;;------------------------------------------------------------------------
;; copied from KSH-MODE .EL 

;; ksh-mode.el --- sh (ksh, bash) script editing mode for GNU Emacs.

;; Copyright (C) 1992-95 Gary Ellison.

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;; LCD Archive Entry:
;; ksh-mode|Gary F. Ellison|Gary_F_Ellison@ATT.COM
;; Mode for editing sh/ksh/bash scripts
;; 23-Feb-95|2.6|~/modes/ksh-mode.el.Z|

;; Author: Gary F. Ellison <Gary.F.Ellison@ATT.COM>
;;                   AT&T Bell Laboratories
;;                   6200 East Broad Street
;;                   Columbus, Ohio 43213 USA
;;
;; Maintainer: Gary F. Ellison <Gary.F.Ellison@ATT.COM>
;; Created: Fri Jun 19
;; Version: 2.6
;; Keywords: languages, shell, korn, bourne, sh, ksh, bash, unix
;;
;; Delta On        : 2/23/95
;; Last Modified By: Gary Ellison
;; Last Modified On: Thu Feb 23 11:32:03 1995
;; Update Count    : 33
;; Status          : Highly Functional
;;

;;; Commentary:

;;
;; Description:
;;   sh, ksh, and bash script editing commands for emacs.
;; 
;; Installation:
;;   Put ksh-mode.el in some directory in your load-path.
;;   Refer to the installation section of ksh-mode's function definition.
;;
;; Usage:
;;   This major mode assists shell script writers with indentation
;;   control and control structure construct matching in much the same
;;   fashion as other programming language modes. Invoke describe-mode
;;   for more information.
;; 
;; Bugs:
;;   When the ksh-align-to-keyword is non-nil and the nester
;;   is a multi-command expression with a compound command
;;   the lines following the compound end will align incorrectly
;;   to the compound command instead of it's current indentation.
;;   The fix will probably require the detection of syntax elements
;;   in the nesting line.
;;   
;;   Function ending brace "}" must be on a separate line for indent-line
;;   to do the right thing.
;;
;;   Explicit function definition matching will proclaim in the minibuffer
;;   "No matching compound command" followed by "Matched ... "
;;
;;   indent-for-comment fails to recognize a comment starting in column 0,
;;   hence it moves the comment-start in comment-column.

;;; Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; HISTORY 
;; 8-Aug-95		Jack Repenning <jackr@sgi.com>
;;    Fix documentation of `ksh-align-to-keyword' to conform to the 23
;;    Feb default change.  Search for keywords obeying case, since the
;;    shell does.
;;
;; 23-Feb-1995		Gary Ellison	
;;    Merged Jonathan Stigelman <Stig@hackvan.com> into 2.5 souce.
;;
;; 23 Feb 1995          Jonathan Stigelman <Stig@hackvan.com>
;;    Reshuffled documentation to make the format more consistant with other
;;    elisp.  Added autoload and removed autoloading instructions from the
;;    ksh-mode docstring.  Changed default value for `ksh-align-to-keyword'
;;    to nil because it doesn't work properly.
;;
;; 2-Aug-1994		Gary Ellison	
;;    Last Modified: Mon Jun 13 16:52:55 1994 #29 (Gary Ellison)
;;    - Syntax table modifications to better support sexp navigation and
;;      parsing.
;;    - Fixed keyword regexps. Keywords were not being recoginized on the
;;      same line as " ' `.
;;
;; 13-Jun-1994		Gary Ellison	
;;    Last Modified: Wed Mar 30 14:12:26 1994 #28 (Gary Ellison)
;;    - Minor excursion problem fixed in ksh-indent-command.
;;
;; 30-Mar-1994		Gary Ellison	
;;    Last Modified: Fri Mar 25 15:42:29 1994 #25 (Gary Ellison)
;;    - Implement user customizable ksh-comment-regexp.
;;    - Make the keyword vs line indentation alignment customizable
;;      by calling ksh-align-to-keyword based on variable of same
;;      name. (If the code is obfuscated or convoluted I can attribute
;;      this to a severe head cold and not malice :)
;;
;; 25-Mar-1994		Gary Ellison	
;;    Last Modified: Fri Feb  4 13:06:30 1994 #23 (Gary Ellison)
;;    - Nest relative to the line indentation not the keywords
;;      column.
;;
;; 4-Feb-1994		Gary Ellison	
;;    Last Modified: Wed Nov 10 10:03:01 1993 #18 (Gary Ellison)
;;    - Add direct support for font-lock-mode. Thanks Espen Skoglund
;;      for the regular expressions.
;;
;; 10-Nov-1993		Gary Ellison	
;;    Last Modified: Tue Oct 12 15:23:06 1993 #17 (Gary Ellison)
;;    Fix message on ksh-match-and-tell to not get invalid format
;;    when a % appears in the string.
;;
;; 12-Oct-1993		Espen Skoglund <espensk@stud.cs.uit.no>.
;;    Last Modified: Tue Oct 12 15:03:01 1993 #16 (Gary Ellison)
;;    Apply Line continuation patch supplied by Espen Skoglund
;;
;; 1-Sep-1993		Gary Ellison	
;;    Last Modified: Tue Aug 17 17:18:18 1993 #14 (Gary Ellison)
;;    Get rid of this-line hack in ksh-get-nester-column.
;;
;; 17-Aug-1993		Gary Ellison	
;;    Last Modified: Mon Jun 21 14:00:43 1993 #13 (Gary Ellison)
;;    Code uses builtin current-indentation instead of lisp defun
;;    ksh-indentation-on-this-line (thanks to Tom Tromey).
;;    More and better doc strings.
;;
;; 5-Aug-1993		Tom Tromey <tromey@cns.caltech.edu>
;;    Last Modified: Thu Aug  5 11:09:12 1993 #12 (Tom Tromey)
;;    ksh-indent-region skips blank lines.  Uses let binding instead
;;    of setq.  No longer marks buffer modified if indentation
;;    doesn't change. 
;;
;; 21-Jun-1993		Gary Ellison	
;;    Last Modified: Mon Mar 29 15:05:34 1993 #11 (Gary Ellison)
;;    Use make-local-variables instead of make-variables-buffer-local
;;    ksh-indent now supports nil (keyword aligned) or number (offset)
;;    Support ksh-tab-always-indent feature
;;    Variables offsetting indentation renamed to better reflect their
;;    role.
;;    Integrate keyword completion feature supplied by
;;    Haavard Rue <hrue@imf.unit.no>.
;;
;; 29-Mar-1993		Gary Ellison	
;;    Last Modified: Tue Sep 29 16:14:02 1992 #10 (Gary Ellison)
;;    Integrate line continuation patch supplied by
;;    Haavard Rue <hrue@imf.unit.no>
;;    Name back to ksh-mode to avoid confusion with sh-mode
;;    by Thomas W. Strong, Jr. <strong+@cmu.edu>.
;;
;; 29-Sep-1992		Gary Ellison	
;;    Last Modified: Wed Sep  2 08:51:40 1992 #9 (Gary Ellison)
;;    Full support of ksh88 case items. 
;;    Align statements under "do" and "then" keywords one position 
;;    past the keyword.
;;
;; 2-Sep-1992		Gary Ellison	
;;    Last Modified: Tue Aug  4 14:34:35 1992 #8 (Gary Ellison)
;;    Use make-variable-buffer-local instead of make-local-variable
;;    Get rid of superflous ksh-default variables.
;;    Use end of word match \b for "then", "do", "else", "elif"
;;    Support process substitution lists and exclude ksh 88 case items
;;    Use default-tab-width for indentation defaults.
;;    Moved installation instructions to the mode level documentation 
;;    section.
;;    Fixed auto-mode-alist documentation.
;;
;; 24-Jul-1992		Gary Ellison	
;;    Last Modified: Fri Jul 24 09:45:11 1992 #7 (Gary Ellison)
;;    Modified ksh-indent-region to use marker versus fixed end point.
;;    comment-start-skip regexp no longer fooled by parameter substitution.
;;    Added constant ksh-mode-version.
;;
;; 21-Jul-1992		Gary Ellison	
;;    Last Modified: Tue Jul 21 15:53:57 1992 #6 (Gary Ellison)
;;    Indent with tabs instead of spaces.
;;    Can handle just about all styles.
;;    Anti-newline in REs.
;;    Word delim "\b" in REs
;;    More syntax entries.
;;    Variables with regexp suffix abbreviated to re
;;    Better } handling
;;    Implemented minimal indent-region-function
;;    Mode documentation corrected.
;;    Minor lisp source format changes.
;;    
;; 29-Jun-1992		Gary Ellison	
;;    Last Modified: Mon Jun 29 15:39:35 1992 #5 (Gary Ellison)
;;    Optimize line-to-string
;;    Implicit/Explicit functions aok
;;    More indentation variables
;;    Superfluous defun killed.
;;    renamed to sh-mode
;;    
;; 22-Jun-1992          Gary Ellison
;;    Last Modified: Mon Jun 22 15:01:14 1992 #4 (Gary Ellison)
;;    Cleanup pre att.emacs posting
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 17:19:14 1992 #3 (Gary Ellison)
;;    Minimal case indent handling
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 16:23:26 1992 #2 (Gary Ellison)
;;    Nesting handled except for case statement
;;
;; 19-Jun-1992          Gary Ellison
;;    Last Modified: Fri Jun 19 10:03:07 1992 #1 (Gary Ellison)
;;    Conception of this mode.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst scl-mode-version "2.6"
  "*Version numbers of this version of scl-mode")

;;
;; Variables controlling indentation style
;;

(defvar scl-indent 3
  ;; perhaps c-basic-offset would be okay to use as a default, but using
  ;; default-tab-width as the default is ridiculous --Stig
  "*Indentation of scl statements with respect to containing block. A value
of nil indicates compound list keyword \(\"do\" and \"then\"\) alignment.")
(defvar scl-case-item-offset scl-indent
  "*Additional indentation for case items within a case statement.")
(defvar scl-case-indent nil
  "*Additional indentation for statements under case items.")
(defvar scl-group-offset (- scl-indent)
  "*Additional indentation for keywords \"do\" and \"then\".")
(defvar scl-brace-offset 0
  "*Additional indentation of \"{\" under functions or brace groupings.")
(defvar scl-multiline-offset 1
  "*Additional indentation of line that is preceded of a line ending with a
\\ to make it continue on next line.")
(defvar scl-match-and-tell t
  "*If non-nil echo in the minibuffer the matching compound command
for the \"done\", \"}\", \"fi\", or \"esac\". ")
(defvar scl-tab-always-indent t
  "*Controls the operation of the TAB key. If t (the default), always
reindent the current line.  If nil, indent the current line only if
point is at the left margin or in the line's indentation; otherwise
insert a tab.")

(defvar scl-align-to-keyword nil
  ;; #### - this is broken, so it should be disabled by default --Stig
  "*Controls whether nested constructs align from the keyword or
the current indentation. If non-nil, indentation will be relative to
the column the keyword starts. If nil, indentation will be relative to
the current indentation of the line the keyword is on.
The default value is nil.
The non-nil case doesn't work very well.")

(defvar scl-comment-regexp "^\\s *#"
  "*Regular expression used to recognize comments. Customize to support
scl-like languages.")

(defun scl-current-indentation ()
  nil
  )
;;
(fset 'scl-current-indentation 'current-column)
;;
;; Variables controlling completion
(defvar scl-completion-list '())
(make-variable-buffer-local 'scl-completion-list)
(set-default 'scl-completion-list  '())

;;
;; -type-  : type number, 0:misc, 1:variable, 2:function
;; -regexp-: regexp used to parse the script
;; -match- : used by match-beginning/end to pickup target
;;
(defvar scl-completion-type-misc 0)
(defvar scl-completion-regexp-var "\\([A-Za-z_0-9]+\\)=")
(defvar scl-completion-type-var 1)
(defvar scl-completion-match-var 1) 
(defvar scl-completion-regexp-var2 "\\$\\({\\|{#\\)?\\([A-Za-z_0-9]+\\)[#%:}]?")
(defvar scl-completion-match-var2 2)
(defvar scl-completion-regexp-function
  "\\(function\\)?[ \t]*\\([A-Za-z_0-9]+\\)[ \t]*([ \t]*)")
(defvar scl-completion-type-function 2)
(defvar scl-completion-match-function 2)

;;
;; Variable controlling fontification
;;
(defvar scl-keywords '("ask" "assert" "discard" "display" "echo" 
"elif" "else" "endfor" "endif" "endwhile" "error" "exit" "for" "global"
"goto" "help" "if" "in" "include" "integer" "log" "off" "on" "pause" "print"
"repeat" "require" "shell" "step" "string" "until" "version" "while"))

;; FIX: add extension keywords for app extensions 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;Core command set 
;
;  ask <prompt> <symbol>
;  assert <condition>
;  discard <symbol>
;  display <...>
;  echo [ ON | OFF ]
;  elif <condition>
;  else
;  endfor
;  endif
;  endwhile
;  exit
;  for <symbol> in <string> ...
;  global [ integer | string ] <symbolname> [ '=' <expression> ]
;  goto <label>
;  help
;  if <condition>
;  include { '-' | <filename> } [ <parm ...]
;  integer <symbolname> [ '=' <expression> ]
;  log [ ON | OFF | <filename> ]
;  on ERROR <label>
;  pause
;  print { <string> | <symbol> } *
;  repeat
;  require [ integer | string ] <symname>
;  shell [ <DOScmd> ]
;  step [ ON | OFF ]
;  string <symname> [ '=' <string> ]
;  until <condition>
;  version
;  while <condition>
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst scl-font-lock-keywords
      (list
       ;; Fontify ( ) expressions
       ;;'("\\([(].*[)]\\)"  1 font-lock-doc-string-face t)

       ;; Fontify keywords
       (cons (concat
	      "\\(\\<"
	      (mapconcat 'identity scl-keywords "\\>\\|\\<")
	      "\\>\\)")
	     1)

       ;; fontify labels 
       '("\\(^[ \t]*[A-Za-z_][A-Za-z_0-9]*:[ \t]*\\)" 1 font-lock-type-face)

       ;; fontify variable declarations
       '("\\<string[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"  1 font-lock-type-face)
       '("\\<integer[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)" 1 font-lock-type-face)
       '("\\<global[ \t]+string[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"   1 font-lock-type-face)
       '("\\<global[ \t]+integer[ \t]+\\([a-zA-Z_][A-Za-z0-9_]*\\)"   1 font-lock-type-face)

       ;; Fontify macros 
       '("\\(\\${.*}\\|\\$[A-Za-z0-9_$]+\\)" 1 font-lock-function-name-face)

       ;; Fontify function names
       ;; '("\\<function[ \t]+\\([^(; \t]+\\)" 1 font-lock-function-name-face)
       ;; '("\\(^[ \t]*[A-Za-z_][A-Za-z_0-9]*[ \t]*()\\)" 1 font-lock-function-name-face)
       ))

(put 'scl-mode	'font-lock-keywords 'scl-font-lock-keywords)

;;
;; Context/indentation regular expressions
;; 
;; indenting expressions
;;

(defconst scl-then-do-re     "^[^#\n]*\\s\"*\\b\\(then\\|do\\)\\b"
  "*Regexp used to locate grouping keywords: \"then\" and \"do\"" )

;;(defconst scl-do-re          "^[ \t]*\\bdo\\(\\b\\|$\\)"
(defconst scl-do-re          "^\\s *\\bdo\\(\\b\\|$\\)"
  "*Regexp used to match keyword: do")

(defconst scl-then-re        "^\\s *\\bthen\\(\\b\\|$\\)"
  "*Regexp used to match keyword: then")

;;
;; Structure starting/indenting keywords
;;
(defconst scl-if-re         "^[^#\n]*\\s\"*\\b\\(if\\)\\b"
  "*Regexp used to match keyword: if")

(defconst scl-for-re        "^[^#\n]*\\s\"*\\b\\(for\\)\\b"
  "*Regexp used to match keyword: for")

(defconst scl-repeat-re     "^[^#\n]*\\s\"*\\b\\(repeat\\)\\b"
  "*Regexp used to match keyword: repeat")

(defconst scl-while-re      "^[^#\n]*\\s\"*\\b\\(while\\)\\b"
  "*Regexp used to match keyword: while")

;;(defconst scl-iteration-keywords-re 
;;  "^[^#\n]*\\s\"*\\b\\(while\\|for\\|until\\|repeat\\)\\b"
;;  "*Match one of the keywords: while, until, for, repeat")

(defconst scl-else-re           "^\\s *\\belse\\(\\b\\|$\\)"
  "*Regexp used to match keyword: else")

(defconst scl-elif-re           "^\\s *\\belif\\(\\b\\|$\\)"
  "*Regexp used to match keyword: elif")

;;(defconst scl-brace-re           "^\\S>*{[ \t\n]"
;;  "*Regexp used to match syntactic entity: { ")
;;
;;(defconst scl-case-item-end-re           "^\\S>*;;[ \t\n]"
;;  "*Regexp used to match case item end syntactic entity: ;;")

(defvar scl-keywords '("ask" "assert" "discard" "display" "echo" 
"elif" "else" "endfor" "endif" "endwhile" "exit" "for" "global"
"goto" "help" "if" "include" "integer" "log" "on" "pause" "print"
"repeat" "require" "shell" "step" "string" "until" "version" "while"))

(defconst scl-keywords-re
  "^[^#\n]*\\s\"*\\b\\(else\\|if\\|elif\\|while\\|repeat\\|for\\|until\\)\\b"
  "*Regexp used to detect compound command keywords: if, else, elif, 
while, for, repeat, and until")


(defconst scl-case-re           "^[^#\n]*\\s\"*\\b\\(case\\)\\b"
  "*Regexp used to match keyword: case")

;;(defconst scl-explicit-func-re
;;  "^\\s *\\(function\\s [a-zA-z_][a-zA-Z0-1_]*\\)\\b"
;;  "*Match an explicit function definition: function name")
;;
;;(defconst scl-implicit-func-re
;;  "^\\s *\\([a-zA-z_][a-zA-Z0-1_]*\\)\\s *()\\s *"
;;  "*Match an implicit function definition: name ()")
;;
;;(defconst scl-func-brace-re "^\\s *\\(.*{\\)[ \t\n]+"
;;  "*Match a implicit function definition brace: name { ")

;;
;; indenting 
(defconst scl-case-item-re           "^[^#\n]*\\s\"*\\()\\)"
  "*Regexp used to match case-items including scl88")

(defconst scl-paren-re           "^[^#\n]*\\s\"*)[ \t\n]+"
  "*Regexp used to match compound list & case items")

;;
;; structure ending keyword regular expressions
(defconst scl-endfor-re          "^\\s *endfor\\b"
  "*Regexp used to match keyword: endfor")

(defconst scl-endif-re            "^\\s *endif\\b"
  "*Regexp used to match keyword: endif")

(defconst scl-until-re            "^\\s *until\\b"
  "*Regexp used to match keyword: until")

(defconst scl-endwhile-re         "^\\s *endwhile\\b"
  "*Regexp used to match keyword: endwhile")

(defconst scl-esac-re          "^\\s *esac\\b"
  "*Regexp used to match keyword: esac")

;;(defconst scl-done-re          "^\\s *done\\b"
;;  "*Regexp used to match keyword: done")
;;
;;(defconst scl-brace-end-re  "^\\s *}\\s *"
;;  "*Regexp used to match function brace-groups")
;;
(defconst scl-multiline-re "^.*\\\\$"
  "*Regexp used to match a line with a statement using more lines.")

;;
;;
;; Create mode specific tables
(defvar scl-mode-syntax-table nil
  "Syntax table used while in scl mode.")
(if scl-mode-syntax-table
    ()
  (setq scl-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\' "\"" scl-mode-syntax-table)
  (modify-syntax-entry ?` "\"" scl-mode-syntax-table)
  (modify-syntax-entry ?\n ">" scl-mode-syntax-table)
  (modify-syntax-entry ?\f ">" scl-mode-syntax-table)
  (modify-syntax-entry ?# "<" scl-mode-syntax-table)
  (modify-syntax-entry ?_ "w" scl-mode-syntax-table)
  (modify-syntax-entry ?< "." scl-mode-syntax-table)
  (modify-syntax-entry ?> "." scl-mode-syntax-table)
  (modify-syntax-entry ?& "." scl-mode-syntax-table)
  (modify-syntax-entry ?| "." scl-mode-syntax-table)
  (modify-syntax-entry ?$ "." scl-mode-syntax-table)
  (modify-syntax-entry ?% "." scl-mode-syntax-table)
  (modify-syntax-entry ?= "." scl-mode-syntax-table)
  (modify-syntax-entry ?/ "." scl-mode-syntax-table)
  (modify-syntax-entry ?+ "." scl-mode-syntax-table)
  (modify-syntax-entry ?* "." scl-mode-syntax-table)
  (modify-syntax-entry ?- "." scl-mode-syntax-table)
  (modify-syntax-entry ?\; "." scl-mode-syntax-table)
  )

; FIX: add ! as keyword 

(defvar scl-mode-abbrev-table nil
  "Abbrev table used while in scl mode.")
(define-abbrev-table 'scl-mode-abbrev-table ())

(defvar scl-mode-map nil 
  "Keymap used in scl mode")

(if scl-mode-map
    ()
  (setq scl-mode-map (make-sparse-keymap))
  (define-key scl-mode-map "\t"    'scl-indent-command)
  (define-key scl-mode-map "\n"    'reindent-then-newline-and-indent)
  (define-key scl-mode-map '[return] 'reindent-then-newline-and-indent)
;;  (define-key scl-mode-map "\t"    'scl-indent-line)
;;  (define-key scl-mode-map "\177"    'backward-delete-char-untabify)
  (define-key scl-mode-map "\C-j"    'reindent-then-newline-and-indent)
  (define-key scl-mode-map "\e\t"    'scl-complete-symbol)
  (define-key scl-mode-map "\C-c\t"    'scl-completion-init-and-pickup)
  )


;;;###autoload
(defun scl-mode ()
  "scl-mode 2.6 - Major mode for editing (Bourne, Korn or Bourne again)
shell scripts.
Special key bindings and commands:
\\{scl-mode-map}
Variables controlling indentation style:
scl-indent
    Indentation of scl statements with respect to containing block.
    Default value is 2.
scl-case-indent
    Additional indentation for statements under case items.
    Default value is nil which will align the statements one position 
    past the \")\" of the pattern.
scl-case-item-offset
    Additional indentation for case items within a case statement.
    Default value is 2.
scl-group-offset
    Additional indentation for keywords \"do\" and \"then\".
    Default value is -2.
scl-brace-offset
    Additional indentation of \"{\" under functions or brace groupings.
    Default value is 0.
scl-multiline-offset
   Additional indentation of line that is preceded of a line ending with a
   \\ to make it continue on next line.
scl-tab-always-indent
    Controls the operation of the TAB key. If t (the default), always
    reindent the current line.  If nil, indent the current line only if
    point is at the left margin or in the line's indentation; otherwise
    insert a tab.
scl-match-and-tell
    If non-nil echo in the minibuffer the matching compound command
    for the \"done\", \"}\", \"fi\", or \"esac\". Default value is t.

scl-align-to-keyword
    Controls whether nested constructs align from the keyword or
    the current indentation. If non-nil, indentation will be relative to
    the column the keyword starts. If nil, indentation will be relative to
    the current indentation of the line the keyword is on.
    The default value is non-nil.

scl-comment-regexp
  Regular expression used to recognize comments. Customize to support
  scl-like languages. Default value is \"\^\\\\s *#\".

Style Guide.
 By setting
    (setq scl-indent default-tab-width)
    (setq scl-group-offset 0)

    The following style is obtained:

    if [ -z $foo ]
	    then
		    bar    # <-- scl-group-offset is additive to scl-indent
		    foo
    fi

 By setting
    (setq scl-indent default-tab-width)
    (setq scl-group-offset (- 0 scl-indent))

    The following style is obtained:

    if [ -z $foo ]
    then
	    bar
	    foo
    fi

 By setting
    (setq scl-case-item-offset 1)
    (setq scl-case-indent nil)

    The following style is obtained:

    case x in *
     foo) bar           # <-- scl-case-item-offset
          baz;;         # <-- scl-case-indent aligns with \")\"
     foobar) foo
             bar;;
    esac

 By setting
    (setq scl-case-item-offset 1)
    (setq scl-case-indent 6)

    The following style is obtained:

    case x in *
     foo) bar           # <-- scl-case-item-offset
           baz;;        # <-- scl-case-indent
     foobar) foo
           bar;;
    esac
    

Installation:
  Put scl-mode.el in some directory in your load-path.
  Put the following forms in your .emacs file.

 (setq auto-mode-alist
      (append auto-mode-alist
              (list
               '(\"\\\\.sh$\" . scl-mode)
               '(\"\\\\.scl$\" . scl-mode)
               '(\"\\\\.bashrc\" . scl-mode)
               '(\"\\\\..*profile\" . scl-mode))))

 (setq scl-mode-hook
      (function (lambda ()
         (font-lock-mode 1)             ;; font-lock the buffer
         (setq scl-indent 8)
	 (setq scl-group-offset -8))
	 (setq scl-brace-offset -8)   
         (setq scl-tab-always-indent t)
         (setq scl-match-and-tell t)
         (setq scl-align-to-keyword t)	;; Turn on keyword alignment
	 )))"
  (interactive)
  (kill-all-local-variables)
  (use-local-map scl-mode-map)
  (setq major-mode 'scl-mode)
  (setq mode-name "Scl")
  (setq local-abbrev-table scl-mode-abbrev-table)
  (set-syntax-table scl-mode-syntax-table)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'scl-indent-line)
  (make-local-variable 'indent-region-function)
  (setq indent-region-function 'scl-indent-region)
  (make-local-variable 'comment-start)
  (setq comment-start "# ")
  (make-local-variable 'comment-end)
  (setq comment-end "")
  (make-local-variable 'comment-column)
  (setq comment-column 32)
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "#+ *")
  ;;
  ;; config font-lock mode
  (make-local-variable 'font-lock-keywords) 
  (setq font-lock-keywords scl-font-lock-keywords)
  ;;
  ;; Let the user customize
  (run-hooks 'scl-mode-hook)
  (if (not scl-align-to-keyword)
      (scl-align-to-keyword -1)
    )
  ) ;; defun

;;
;; Support functions

(defun scl-align-to-keyword (&optional arg)
  "Toggle value of scl-align-to-keyword and rebind the scl-current-indentation
function. With arg, force alignment to keyword if and only if arg is positive."
  (interactive)
  (if (null arg)			;just toggle
      (cond ((not scl-align-to-keyword)
	     (setq scl-align-to-keyword t)
	     (fset 'scl-current-indentation 'current-column))
	    (t
	     (setq scl-align-to-keyword nil)
	     (fset 'scl-current-indentation 'current-indentation))
	    )
    (cond ((natnump arg)
	   (setq scl-align-to-keyword t)
	   (fset 'scl-current-indentation 'current-column))
	  (t
	   (setq scl-align-to-keyword nil)
	   (fset 'scl-current-indentation 'current-indentation))
	  ))
  )

(defun scl-current-line ()
  "Return the vertical position of point in the buffer.
Top line is 1."
  (+ (count-lines (point-min) (point))
     (if (= (current-column) 0) 1 0))
  )


(defun scl-line-to-string ()
  "From point, construct a string from all characters on
current line"
  (skip-chars-forward " \t") ;; skip tabs as well as spaces
  (buffer-substring (point)
                    (progn
                      (end-of-line 1)
                      (point))))

(defun scl-get-nest-level ()
  "Return a 2 element list (nest-level nest-line) describing where the
current line should nest."
  (let ((case-fold-search)
    	(level))
    (save-excursion
      (forward-line -1)
      (while (and (not (bobp))
		  (null level))
	(if (and (not (looking-at "^\\s *$"))
 		 (not (save-excursion
 			(forward-line -1)
 			(beginning-of-line)
			(looking-at scl-multiline-re)))
		 (not (looking-at scl-comment-regexp)))
	    (setq level (cons (current-indentation)
			      (scl-current-line)))
	  (forward-line -1)
	  );; if
	);; while
      (if (null level)
	  (cons (current-indentation) (scl-current-line))
	level)
      )
    )
  )

(defun scl-looking-at-compound-list ()
  "Return true if current line contains compound list initiating keyword"
  (or 
   (looking-at scl-do-re)
   (looking-at scl-then-re)
   ) ;; or
  ) ;; defun

(defun scl-looking-at-case-item ()
  "Return true if current line is a case-item .vs. paren compound list"
  nil)

;(defun scl-looking-at-case-item ()
;  "Return true if current line is a case-item .vs. paren compound list"
;  (save-excursion
;    (beginning-of-line)
;    ;;
;    ;; Handle paren indentation constructs for this line
;    (cond ((looking-at scl-paren-re)
;	   (goto-line (cdr (scl-get-nest-level)))
;	   ;;
;	   ;; The question is whether this is really a case item or just
;	   ;; parenthesized compound list.
;	   (cond ((or (looking-at scl-case-re)
;		      (looking-at scl-case-item-end-re)))
;		 ;;
;		 ;; turns out to be a parenthesized compound list
;		 ;; so propigate the nil for cond
;		 )
;	   ))
;    )
;  );; defun

(defun scl-get-case-indent ()
  "Return the column of the closest open case statement"
  (save-excursion
    (let (
	  (nest-list (scl-get-compound-level scl-case-re scl-esac-re (point)))
	  )
      (if (null nest-list)
	  (progn 
	    (if scl-match-and-tell
		(message "No matching case for ;;"))
	    0)
	(car nest-list)))
    )
  )

;;
;; Functions which make this mode what it is
;;

(defun scl-get-nester-column (nest-line)
  "Return the column to indent to with respect to nest-line taking 
into consideration keywords and other nesting constructs."
  (save-excursion 
    (let ((fence-post)
	  (nester-column)
	  (case-fold-search)
	  (start-line (scl-current-line)))
      ;;
      ;; Handle case item indentation constructs for this line
      (cond ((scl-looking-at-case-item)
	     (save-excursion
	       (goto-line nest-line)
	       (let ((fence-post (save-excursion (end-of-line) (point))))
		 ;;
		 ;; Now know there is a case-item so detect whether
		 ;; it is first under case, just another case-item, or
		 ;; a case-item and case-item-end all rolled together.
		 ;;
		 (cond ((re-search-forward scl-case-re fence-post t)
			(goto-char (match-beginning 1))
			(+ (scl-current-indentation) scl-case-item-offset))

		       ((scl-looking-at-case-item)
			(current-indentation))

		       ;;((looking-at scl-case-item-end-re)
		       ;; (end-of-line)
		       ;;  (+ (scl-get-case-indent) scl-case-item-offset))
		       )
		 )))
	    (t;; Not a case-item.  What to do relative to the nest-line?
	     (save-excursion
	       (goto-line nest-line)
	       (setq fence-post (save-excursion (end-of-line) (point)))
	       (setq nester-column
		     (save-excursion
		       (cond
			;;
			;; Check if we are in a continued statement
			((and (looking-at scl-multiline-re)
			      (save-excursion
				(goto-line (1- start-line))
				(looking-at scl-multiline-re)))
			 (+ (current-indentation) scl-multiline-offset))

			;; In order to locate the column of the keyword,
			;; which might be embedded within a case-item,
			;; it is necessary to use re-search-forward.
			;; Search by literal case, since shell is
			;; case-sensitive.
			((re-search-forward scl-keywords-re fence-post t)
			 (goto-char (match-beginning 1))
			 (if (looking-at scl-case-re)
			     (+ (scl-current-indentation) scl-case-item-offset)
			   (+ (scl-current-indentation)
			      (if (null scl-indent)
				  2 scl-indent)
			      )))

			((re-search-forward scl-then-do-re fence-post t)
			 (if (null scl-indent)
			     (progn 
			       (goto-char (match-end 1))
			       (+ (scl-current-indentation) 1))
			   (progn
			     (goto-char (match-beginning 1))
			     (+ (scl-current-indentation) scl-indent))
			   ))

			;;((looking-at scl-brace-re)
			;; (+ (current-indentation)
			;;    (if (null scl-indent)
		        ;;	2 scl-indent)
			;;    ))

			;;
			;; Forces functions to first column
			;;((or (looking-at scl-implicit-func-re)
			;;     (looking-at scl-explicit-func-re))
			;; (if (looking-at scl-func-brace-re)
			;;     (if (null scl-indent)
			;;	 2 scl-indent)
			;;   scl-brace-offset))

			;;
			;; Need to first detect the end of a case-item
			;;((looking-at scl-case-item-end-re)
			;; (end-of-line)
			;; (+ (scl-get-case-indent) scl-case-item-offset))
			;;
			;; Now detect first statement under a case item
			((scl-looking-at-case-item)
			 (if (null scl-case-indent)
			     (progn
			       (re-search-forward scl-case-item-re fence-post t)
			       (goto-char (match-end 1))
			       (+ (current-column) 1))
			   (+ (current-indentation) scl-case-indent)))

			;; This is hosed when using current-column
			;; and there is a multi-command expression as the
			;; nester.
			(t (current-indentation)))
		       )
		     ));; excursion over
	     ;;
	     ;; Handle additional indentation constructs for this line
	     (cond ((scl-looking-at-compound-list)
		    (+ nester-column scl-group-offset))
		   ;;((looking-at scl-brace-re)
		   ;; (+ nester-column scl-brace-offset))
		   (t nester-column))
	     );; Not a case-item
	    )
      );;let
    );; excursion
  );; defun

(defun scl-indent-command ()
  "Indent current line relative to containing block and allow for
scl-tab-always-indent customization"
  (interactive)
  (let (case-fold-search)
    (cond ((save-excursion
	     (skip-chars-backward " \t")
	     (bolp))
	   (scl-indent-line))
	  (scl-tab-always-indent
	   (save-excursion
	     (scl-indent-line)))
	  (t (insert-tab))
	  ))
  )


(defun scl-indent-line ()
  "Indent current line as far as it should go according
to the syntax/context"
  (interactive)
  (let (case-fold-search)
    (save-excursion
      (beginning-of-line)
      (if (bobp)
	  nil
	;;
	;; Align this line to current nesting level
	(let*
	    (
	     (level-list (scl-get-nest-level)) ; Where to nest against
	     ;;           (last-line-level (car level-list))
	     (this-line-level (current-indentation))
	     (nester-column (scl-get-nester-column (cdr level-list)))
	     (struct-match (scl-match-structure-and-reindent))
	     )
	  (if struct-match
	      (setq nester-column struct-match))
	  (if (eq nester-column this-line-level)
	      nil
	    (beginning-of-line)
	    (let ((beg (point)))
	      (back-to-indentation)
	      (delete-region beg (point)))
	    (indent-to nester-column))
	  );; let*
	);; if
      );; excursion
    ;;
    ;; Position point on this line
    (let*
	(
	 (this-line-level (current-indentation))
	 (this-bol (save-excursion
		     (beginning-of-line)
		     (point)))
	 (this-point (- (point) this-bol))
	 )
      (cond ((> this-line-level this-point);; point in initial white space
	     (back-to-indentation))
	    (t nil)
	    );; cond
      );; let*
    );; let
  );; defun


(defun scl-match-indent-level (begin-re end-re)
  "Match the compound command and indent. Return nil on no match,
indentation to use for this line otherwise."
  (interactive)
  (let* ((case-fold-search)
	 (nest-list 
	  (save-excursion
	    (scl-get-compound-level begin-re end-re (point))
	    ))
	 ) ;; bindings
    (if (null nest-list)
	(progn
	  (if scl-match-and-tell
	      (message "No matching compound command"))
	  nil) ;; Propagate a miss.
      (let* (
	     (nest-level (car nest-list))
	     (match-line (cdr nest-list))
	     ) ;; bindings
	(if scl-match-and-tell
	    (save-excursion
	      (goto-line match-line)
	      (message "Matched ... %s" (scl-line-to-string))
	      ) ;; excursion
	  ) ;; if scl-match-and-tell
	nest-level ;;Propagate a hit.
	) ;; let*
      ) ;; if
    ) ;; let*
  ) ;; defun scl-match-indent-level

(defun scl-match-structure-and-reindent ()
  "If the current line matches one of the indenting keywords
or one of the control structure ending keywords then reindent. Also
if scl-match-and-tell is non-nil the matching structure will echo in
the minibuffer"
  (interactive)
  (let (case-fold-search)
    (save-excursion
      (beginning-of-line)
      (cond ((looking-at scl-else-re)
	     (scl-match-indent-level scl-if-re scl-endif-re))
	    ((looking-at scl-elif-re)
	     (scl-match-indent-level scl-if-re scl-endif-re))
	    ((looking-at scl-endif-re)
	     (scl-match-indent-level scl-if-re scl-endif-re))
	    ((looking-at scl-endwhile-re)
	     (scl-match-indent-level scl-while-re scl-endwhile-re))
	    ((looking-at scl-until-re)
	     (scl-match-indent-level scl-repeat-re scl-until-re))
	    ((looking-at scl-endfor-re)
	     (scl-match-indent-level scl-for-re scl-endfor-re))

	    ;;((looking-at scl-done-re)
	    ;; (scl-match-indent-level scl-iteration-keywords-re scl-done-re))
	    ;;((looking-at scl-esac-re)
	    ;; (scl-match-indent-level scl-case-re scl-esac-re))
	    ;;
	    ;;((looking-at scl-brace-end-re)
	    ;; (cond
	    ;;  ((scl-match-indent-level scl-implicit-func-re scl-brace-end-re))
	    ;;  ((scl-match-indent-level scl-explicit-func-re scl-brace-end-re))
	    ;;  ((scl-match-indent-level scl-func-brace-re scl-brace-end-re))
	    ;;  (t nil)))
	    (t nil)
	    );; cond
      )
    ))

(defun scl-get-compound-level 
  (begin-re end-re anchor-point &optional balance-list)
  "Determine how much to indent this structure. Return a list (level line) 
of the matching compound command or nil if no match found."
  (let* 
      (;; Locate the next compound begin keyword bounded by point-min
       (match-point (if (re-search-backward begin-re (point-min) t)
			(match-beginning 1) 0))
       (nest-column (if (zerop match-point)
			1 
		      (progn
			(goto-char match-point)
			(scl-current-indentation))))
       (nest-list (cons 0 0))    ;; sentinel cons since cdr is >= 1
       )
    (if (zerop match-point)
	nil ;; graceful exit from recursion
      (progn
	(if (nlistp balance-list)
	    (setq balance-list (list)))
	;; Now search forward from matching start keyword for end keyword
	(while (and (consp nest-list) (zerop (cdr nest-list))
		    (re-search-forward end-re anchor-point t))
	  (if (not (memq (point) balance-list))
	      (progn
		(setq balance-list (cons (point) balance-list))
		(goto-char match-point)  ;; beginning of compound cmd
		(setq nest-list
		      (scl-get-compound-level begin-re end-re
					     anchor-point balance-list))
		)))

	(cond ((consp nest-list)
	       (if (zerop (cdr nest-list))
		 (progn
		   (goto-char match-point)
		   (cons nest-column (scl-current-line)))
		 nest-list))
	      (t nil)
	      )
	)
      )
    )
  )


(defun scl-indent-region (start end)
  "From start to end, indent each line."
  ;; The algorithm is just moving through the region line by line with
  ;; the match noise turned off.  Only modifies nonempty lines.
  (save-excursion
    (let (scl-match-and-tell
	  (endmark (copy-marker end)))
      
      (goto-char start)
      (beginning-of-line)
      (setq start (point))
      (while (> (marker-position endmark) start)
	(if (not (and (bolp) (eolp)))
	    (scl-indent-line))
	(forward-line 1)
	(setq start (point)))

      (set-marker endmark nil)
      )
    )
  )

;;
;; Completion code supplied by Haavard Rue <hrue@imf.unit.no>.
;;
;;
;; add a completion with a given type to the list
;;
(defun scl-addto-alist (completion type)
  (setq scl-completion-list
	(append scl-completion-list
		(list (cons completion type)))))
;;
;; init the list and pickup all 
;;
(defun scl-completion-init-and-pickup ()
  (interactive)
  (let (case-fold-search)
    (scl-completion-list-init)
    (scl-pickup-all)))

;;
;; init the list
;;

;; FIX: fix completion list to match SCL keywords 
;

(defun scl-completion-list-init ()
  (interactive)
  (setq scl-completion-list
	(list
	 (cons "ask"  scl-completion-type-misc)
	 (cons "assert"  scl-completion-type-misc)
	 (cons "discard"  scl-completion-type-misc)
	 (cons "display"  scl-completion-type-misc)
	 (cons "echo"  scl-completion-type-misc)
	 (cons "elif"  scl-completion-type-misc)
	 (cons "else"  scl-completion-type-misc)
	 (cons "endfor"  scl-completion-type-misc)
	 (cons "endif"  scl-completion-type-misc)
	 (cons "endwhile"  scl-completion-type-misc)
	 (cons "error"  scl-completion-type-misc)
	 (cons "exit"  scl-completion-type-misc)
	 (cons "for"  scl-completion-type-misc)
	 (cons "global"  scl-completion-type-misc)
	 (cons "goto"  scl-completion-type-misc)
	 (cons "help"  scl-completion-type-misc)
	 (cons "if"  scl-completion-type-misc)
	 (cons "in"  scl-completion-type-misc)
	 (cons "include"  scl-completion-type-misc)
	 (cons "integer"  scl-completion-type-misc)
	 (cons "log"  scl-completion-type-misc)
	 (cons "off"  scl-completion-type-misc)
	 (cons "on"  scl-completion-type-misc)
	 (cons "pause"  scl-completion-type-misc)
	 (cons "print"  scl-completion-type-misc)
	 (cons "repeat"  scl-completion-type-misc)
	 (cons "require"  scl-completion-type-misc)
	 (cons "shell"  scl-completion-type-misc)
	 (cons "step"  scl-completion-type-misc)
	 (cons "string"  scl-completion-type-misc)
	 (cons "untill"  scl-completion-type-misc)
	 (cons "version"  scl-completion-type-misc)
	 (cons "while"  scl-completion-type-misc)

;	 (cons "if"  scl-completion-type-misc)
;	 (cons "while"  scl-completion-type-misc)
;	 (cons "until"  scl-completion-type-misc)
;	 (cons "select"  scl-completion-type-misc)
;	 (cons "for"  scl-completion-type-misc)
;	 (cons "continue"  scl-completion-type-misc)
;	 (cons "function"  scl-completion-type-misc)
;	 (cons "fi"  scl-completion-type-misc)
;	 (cons "case"  scl-completion-type-misc)
;	 (cons "esac"  scl-completion-type-misc)
;	 (cons "break"  scl-completion-type-misc)
;	 (cons "exit"  scl-completion-type-misc)
;	 (cons "done"  scl-completion-type-misc)
;	 (cons "do"  scl-completion-type-misc)
          )))

(defun scl-eol-point ()
  (save-excursion
    (end-of-line)
    (point)))

(defun scl-bol-point ()
  (save-excursion
    (beginning-of-line)
    (point)))

(defun scl-pickup-all ()
  "Pickup all completions in buffer."
  (interactive)
  (scl-pickup-completion-driver (point-min) (point-max) t))

(defun scl-pickup-this-line ()
  "Pickup all completions in current line."
  (interactive)
  (scl-pickup-completion-driver (scl-bol-point) (scl-eol-point) nil))

(defun scl-pickup-completion-driver (pmin pmax message)
  "Driver routine for scl-pickup-completion."
  (if message
      (message "pickup completion..."))
  (let* (
	 (i1
	  (scl-pickup-completion  scl-completion-regexp-var
				 scl-completion-type-var
				 scl-completion-match-var
				 pmin pmax))
	 (i2
	  (scl-pickup-completion  scl-completion-regexp-var2
				 scl-completion-type-var
				 scl-completion-match-var2
				 pmin pmax))
	 (i3
	  (scl-pickup-completion  scl-completion-regexp-function
				 scl-completion-type-function
				 scl-completion-match-function
				 pmin pmax)))
    (if message
	(message "pickup %d variables and %d functions." (+ i1 i2) i3))))

(defun scl-pickup-completion (regexp type match pmin pmax)
  "Pickup completion in region and addit to the list, if not already
there." 
  (let ((i 0) kw obj)
    (save-excursion
      (goto-char pmin)
      (while (and
	      (re-search-forward regexp pmax t)
	      (match-beginning match)
	      (setq kw  (buffer-substring
			 (match-beginning match)
			 (match-end match))))
	(progn
	  (setq obj (assoc kw scl-completion-list))
	  (if (or (equal nil obj)
		  (and (not (equal nil obj))
		       (not (= type (cdr obj)))))
	      (progn
		(setq i (1+ i))
		(scl-addto-alist kw type))))))
    i))

(defun scl-complete-symbol ()
  "Perform completion."
  (interactive)
  (let* ((case-fold-search)
	 (end (point))
         (beg (unwind-protect
                  (save-excursion
                    (backward-sexp 1)
                    (while (= (char-syntax (following-char)) ?\')
                      (forward-char 1))
                    (point))))
         (pattern (buffer-substring beg end))
	 (predicate 
	  ;;
	  ;; ` or $( mark a function
	  ;;
	  (save-excursion
	    (goto-char beg)
	      ;;
	      ;; a $, or ${ mark a variable
	      ;;
	      (if (or
		   (save-excursion
		     (backward-char 1)
		     (looking-at "\\$"))
		   (save-excursion
		     (backward-char 2)
		     (looking-at "\\${")))
		  (function (lambda (sym)
			      (equal (cdr sym)
				     scl-completion-type-var)))
		;;
		;; don't know. use 'em all
		;;
		(function (lambda (sym) t)))))
	 ;;
	 (completion (try-completion pattern scl-completion-list predicate)))
    ;;
    (cond ((eq completion t))
	  ;;
	  ;; oops, what is this ?
	  ;;
          ((null completion)
           (message "Can't find completion for \"%s\"" pattern))
	  ;;
	  ;; insert
	  ;;
          ((not (string= pattern completion))
           (delete-region beg end)
           (insert completion))
	  ;;
	  ;; write possible completion in the minibuffer,
	  ;; use this instead of a seperate buffer (usual)
	  ;;
          (t
           (let ((list (all-completions pattern scl-completion-list predicate))
		 (string ""))
	     (while list
	       (progn
		 (setq string (concat string (format "%s " (car list))))
		 (setq list (cdr list))))
	     (message string))))))

(provide 'scl-mode)
;;; scl-mode.el ends here
